(ns prism.postgres2
  (:require
    [clojure.core.cache.wrapped :as ccw]
    [honey.sql :as hsql]
    [next.jdbc :as jdbc]
    [next.jdbc.connection :as connection]
    [next.jdbc.result-set :refer [as-unqualified-kebab-maps]]
    [prism.core :refer [defdelayed] :as prism]
    [prism.internal.pg-extensions :as pge])
  (:import
    (com.zaxxer.hikari HikariDataSource)
    (java.sql Connection)))

(pge/extend-protocols!)

(def ^:dynamic ^Connection *connections* {})
(def ^:dynamic ^Connection *transactions* {})

(defprotocol SqlClient
  (^Connection connect [this])
  (format-sql [this statement options])
  (execute! [this statement options])
  (execute-batch! [this statement params options])
  (exists? [this statement options])
  (rollback-txn! [this]))

(defn kw->param [kw]
  (keyword (str \? (name kw))))

(defmacro ->params [& kws]
  (into {} (map (fn [kw] [kw (kw->param kw)])) kws))

(defn updated? [result]
  (-> result prism/vec-first :next.jdbc/update-count (> 0)))

(defn ->data-source [{:keys [pool]
                      :as   postgres-config}]
  (->> (assoc postgres-config :dbtype "postgres")
       connection/jdbc-url
       (assoc pool :jdbcUrl)
       (connection/->pool HikariDataSource)))

(defmacro with-conn [client & body]
  `(let [client# ~client]
     (with-open [c# (connect client#)]
       (binding [*connections* (assoc *connections* client# c#)]
         ~@body))))

(defmacro with-txn [client & body]
  `(let [client# ~client]
     (with-open [c# (connect client#)]
       (jdbc/with-transaction
         [~'txn c#]
         (binding [*connections* (assoc *connections* client# c#)
                   *transactions* (assoc *transactions* client# ~'txn)]
           ~@body)))))

(defrecord PostgresClient [data-src client-config]
  SqlClient
  (connect [_]
    (jdbc/get-connection data-src))
  (format-sql [{{:keys [format-options query-cache]} :client-config} statement options]
    (let [params (dissoc options ::cached?)
          opts (cond-> format-options
                       (seq params) (assoc :params params)
                       (and query-cache
                            (::cached? options)) (assoc :cache query-cache))]
      (hsql/format statement opts)))
  (execute! [this statement {:keys [::return-keys] :as options}]
    (if-let [c (get *connections* this)]
      (jdbc/execute!
        c
        (format-sql this statement options)
        (cond-> {:builder-fn as-unqualified-kebab-maps}
                (seq return-keys) (assoc :return-keys (mapv (comp prism/vec-first hsql/format-expr) return-keys))))
      (throw (IllegalStateException. "This client does not have an open connection."))))
  (execute-batch! [this statement params opts]
    (when (seq params)
      (if-let [c (get *connections* this)]
        (let [params-template (-> (first params)
                                  (update-vals (constantly 0)))
              ordered-params (mapv
                               (apply juxt (keys params-template))
                               params)
              statement (-> (hsql/format statement {:params params-template})
                            (subvec 0 1))]
          (with-open [ps (jdbc/prepare c statement opts)]
            (jdbc/execute-batch! ps ordered-params opts)))
        (throw (IllegalStateException. "This client does not have an open connection.")))))
  (exists? [this statement options]
    (-> (execute! this
          {:select [[true :exists]]
           :where  [:exists (assoc statement :select [1])]}
          options)
        first
        :exists))
  (rollback-txn! [this]
    (if-let [^Connection txn (get *transactions* this)]
      (.rollback txn)
      (throw (IllegalStateException. "This client does not have an open transaction.")))))

(defn default-client-query-cache []
  (ccw/soft-cache-factory {}))

(defdelayed ^HikariDataSource default-client
  (-> (prism/config)
      :postgres
      ->data-source
      (->PostgresClient {:query-cache (default-client-query-cache)})))

(defdelayed ^HikariDataSource default-greptime-client
  (-> (prism/config)
      :greptime
      ->data-source
      (->PostgresClient {:format-options {:inline true}
                         :query-cache    (default-client-query-cache)})))

(comment
  (with-txn (default-greptime-client)
    (execute! (default-greptime-client)
      [:raw "select * from numbers limit 5;"] {})))
