(ns prism.s3
  (:require
    [clojure.data.xml :as xml]
    [clojure.java.io :as io]
    [hato.middleware :as hmw]
    [prism.aws :as aws]
    [prism.core :refer [defdelayed] :as prism])
  (:import
    (java.nio.charset StandardCharsets)
    (java.security MessageDigest)
    (java.util Base64)))

(def aws-service-name "s3")

(defn- s3-path
  ([bucket] (str "/" bucket))
  ([bucket key] (str (s3-path bucket) "/" key)))

(defprotocol S3Client
  (list-objects [this request] [this request http-config])
  (delete-object! [this request] [this request http-config])
  (put-object! [this request] [this request http-config])
  (get-object [this request] [this request http-config])
  (get-object-body [this request] [this request http-config])
  (get-object-metadata [this request] [this request http-config])
  (delete-objects! [this request] [this request http-config]))

(defn- delete-objects-request ^String [ks quiet?]
  (let [base (cond-> [:Delete {:xmlns "http://s3.amazonaws.com/doc/2006-03-01/"}]
                     quiet? (conj [:Quiet "true"]))]
    (->> (into base
               (map (fn [k] [:Object [:Key k]]))
               ks)
         xml/sexp-as-element
         xml/emit-str)))

(defn- aws-md5-header [body-bytes]
  (let [md (MessageDigest/getInstance "MD5")]
    (.update md ^bytes body-bytes)
    (-> (.encode (Base64/getEncoder) (.digest md))
        (String. StandardCharsets/US_ASCII))))

(defrecord BasicS3Client [aws-client]
  S3Client
  (list-objects [this req] (list-objects this req {}))
  (list-objects [_ {:keys [bucket delimiter prefix max-keys continuation-token]
                    :or   {delimiter "/"}} http-config]
    (aws/request
      aws-client
      (s3-path bucket)
      (assoc
        http-config
        :method :get
        :query-params (cond-> {"list-type" "2"
                               "delimiter" delimiter
                               "prefix"    prefix}
                              continuation-token (assoc "continuation-token" continuation-token)
                              max-keys (assoc "max-keys" max-keys))
        :as :xml)))
  (delete-object! [this req] (delete-object! this req {}))
  (delete-object! [_ {:keys [bucket key]} http-config]
    (aws/request
      aws-client
      (s3-path bucket key)
      (assoc http-config :method :delete)))
  (put-object! [this req] (put-object! this req {}))
  (put-object! [_ {:keys [bucket key body]} http-config]
    (aws/request
      aws-client
      (s3-path bucket key)
      (assoc http-config
        :method :put
        :body body)))
  (get-object [this req] (get-object this req {}))
  (get-object [_ {:keys [bucket key]} http-config]
    (aws/request
      aws-client
      (s3-path bucket key)
      (-> (update http-config :as (fnil identity :stream))
          (assoc :method :get))))
  (get-object-body [this req] (get-object-body this req {}))
  (get-object-body [this request http-config]
    (let [{:keys [status body]} (get-object this request (assoc http-config :throw-exceptions false))]
      (cond
        (hmw/unexceptional-status? status) body
        (= 404 status) nil
        :else (throw (ex-info "Failed to fetch object" (assoc request :status status :body (slurp body)))))))
  (get-object-metadata [this req] (get-object-metadata this req {}))
  (get-object-metadata [_ {:keys [bucket key]} http-config]
    (aws/request
      aws-client
      (s3-path bucket key)
      (assoc http-config
        :method :head
        :as :stream)))
  (delete-objects! [this req] (delete-objects! this req {}))
  (delete-objects! [this {:keys [bucket quiet?] :as opts} http-config]
    (loop [responses []
           continuation-token nil]
      (let [opts (cond-> opts
                         continuation-token (assoc :continuation-token continuation-token))
            {:keys [status]
             :as   list-response} (list-objects this opts http-config)]
        (if (hmw/unexceptional-status? status)
          (conj responses list-response)
          (let [result (-> list-response :body :ListBucketResult)
                keys (as-> (:Contents result) $
                           (if (map? $) (vector $) $)
                           (mapv :Key $))
                {:keys [status]
                 :as   delete-response} (when (seq keys)
                                          (as-> (.getBytes (delete-objects-request keys quiet?)) body
                                                (aws/request
                                                  aws-client
                                                  (str (s3-path bucket) "/?delete")
                                                  (-> (assoc http-config :method :post
                                                                         :as :xml
                                                                         :body body)
                                                      (assoc-in [:headers "Content-MD5"] (aws-md5-header body))))))]
            (cond
              (hmw/unexceptional-status? status) (cond-> (conj responses delete-response)
                                                         (:IsTruncated result) (recur (:NextContinuationToken result)))
              (nil? delete-response) responses
              :else (conj responses delete-response))))))))

(defdelayed default-s3-client
  (let [{{:keys [username password region]
          :or   {region "us-east-1"}} :s3} (prism/config)]
    (-> (aws/create-client
          ::aws/service
          {:access-key     username
           :secret-key     password
           :region         region
           :service        aws-service-name
           :consul-service :s3})
        ->BasicS3Client)))

(comment
  (def webhook-s3-client (-> (aws/create-client
                               ::aws/static-url
                               {:region  "ap-southeast-2"
                                :service aws-service-name
                                :url     "http://localhost:8084/62a7b50b-e9ee-45ac-aee1-0a6fa1a970aa"})
                             ->BasicS3Client))
  (let [filename "app-latest.tar.gz"]
    (put-object!
      (default-s3-client)
      {:bucket      "test"
       :key         filename
       :http-client (prism.http/->DefaultHatoClient)
       :body        (io/input-stream (io/file (str "s3-tests/" filename)))}
      {:timeout 100000}))
  (put-object!
    webhook-s3-client
    {:bucket "test"
     :key    "some-key"
     :body   (io/input-stream (io/file "/Users/dekelpilli/Downloads/CSV_Data_2024_12_2 15_26.csv"))}
    {:async? true})
  (put-object!
    (-> (aws/create-client
          ::aws/static-url
          {:region  "ap-southeast-2"
           :service aws-service-name
           :url     "http://localhost:1234"})
        ->BasicS3Client)
    {:bucket "test"
     :key    "some-key"
     :body   (io/input-stream (io/file "deps.edn"))}
    {})
  (get-object-metadata
    {:bucket "test"
     :key    "app-latest.tar.gz"}
    :timeout 100000)
  (list-objects
    (default-s3-client)
    {:bucket   "test"
     :max-keys 2}
    {})
  (list-objects
    aws-s3-client
    {:bucket "temp-test-replay"}
    {:throw-exceptions false})
  (get-object-metadata {:bucket "test"
                        :key    "some-key"})
  (get-object {:bucket "test"
               :key    "response_test.txt"})

  (delete-objects! {:prefix "some-"
                    :quiet? true
                    :bucket "test"}))
