(ns prism.ring-http
  (:require
    [clojure.string :as s]
    [malli.core :as m]
    [malli.transform :as mt]
    [malli.util :as mu]
    [muuntaja.core :as muuntaja]
    [muuntaja.interceptor :as mi]
    [prism.core :as prism]
    [prism.http :as http]
    [prism.internal.classpath :as cp]
    [reitit.coercion :as coercion]
    [reitit.coercion.malli :as mc]
    [reitit.core :as r]
    [reitit.http :as rhttp]
    [reitit.http.coercion :as rhc]
    [reitit.http.interceptors.exception :as exception]
    [reitit.http.interceptors.parameters :as parameters]
    [reitit.impl :as ri]
    [reitit.interceptor.sieppari :as sieppari]
    [reitit.ring :as ring]
    [reitit.spec :as rs]
    [ring.core.protocols :as rcp])
  (:import
    (clojure.lang ExceptionInfo Seqable)
    (java.io ByteArrayOutputStream IOException InputStream OutputStream)
    (java.net.http HttpTimeoutException)
    (java.time Instant)))

(def muuntaja-instance (cp/if-ns 'prism.json
                         (-> (update muuntaja/default-options :formats assoc "application/json" prism.json/muuntaja-format)
                             muuntaja/create)
                         muuntaja/instance))

(defn- reference-expand [endpoints-registry]
  (fn [data opts]
    (if (keyword? data)
      (some-> data
              endpoints-registry
              (r/expand opts)
              (assoc :name data))
      (r/expand data opts))))

(defn- format-exception [e]
  (let [cause (when-let [cause (ex-cause e)]
                (format-exception cause))
        info (if (instance? ExceptionInfo e)
               {:message (ex-message e)
                :data    (ex-data e)}
               {:type    (.getName (class e))
                :message (ex-message e)})]
    (cond-> info
            cause (assoc :cause cause))))

(comment
  (format-exception
    (Exception. "top"
                (Exception. "middle"
                            (Exception. "root")))))

(defn- exception-handler [message exception request]
  {:status 500
   :body   {:message   message
            :exception (format-exception exception)
            :uri       (:uri request)}})

(defn- exception-wrapper [handler e request]
  (handler e request))

(cp/when-ns 'taoensso.timbre
  (defn- exception-wrapper [handler e request]
    (let [context (-> (select-keys request [:request-method :scheme :content-length :content-type :uri
                                            :parameters :query-params :path-params :query-string :body-params])
                      (update-in [:parameters :header] dissoc :authorization))
          {ex-type :type
           :as     ex-data} (ex-data e)]
      (case ex-type
        :reitit.coercion/request-coercion (taoensso.timbre/with-context
                                            context
                                            (taoensso.timbre/debugf "Request coercion error in '%s'." (:in ex-data)))
        :reitit.coercion/response-coercion (taoensso.timbre/with-context
                                             (assoc context :response (:response ex-data))
                                             (taoensso.timbre/error "Response coercion error."))
        (:ring.util.http-response/response ::ring/response) nil
        (taoensso.timbre/with-context
          context
          (taoensso.timbre/errorf e "Unexpected exception while handling HTTP request."))))
    (handler e request)))

(defn- avoid-class-error-type [e]
  (if (-> (ex-data e) :type class?)
    (ex-info
      (ex-message e)
      (assoc (ex-data e)
        :type ::exception/default
        :class (-> (ex-data e) :type))
      (ex-cause e))
    e))

(defn exception-interceptor []
  (-> (merge
        exception/default-handlers
        {::error                           (partial exception-handler "error")
         ::exception                       (partial exception-handler "exception")
         :ring.util.http-response/response exception/http-response-handler
         ::exception/default               (partial exception-handler "default")
         ::exception/wrap                  exception-wrapper
         HttpTimeoutException              (fn [_e _req] {:status 504})
         IOException                       (fn [e req]
                                             (if (some-> (ex-message e)
                                                         (s/starts-with? "Received RST_STREAM"))
                                               (do (taoensso.timbre/with-context {:request req}
                                                     (taoensso.timbre/debugf e "Request was cancelled by client."))
                                                   {:status 499})
                                               (exception-handler "exception" e req)))})
      exception/exception-interceptor
      (update :error
              (fn [interceptor-fn]
                (fn [ctx]
                  (-> (update ctx :error avoid-class-error-type)
                      interceptor-fn))))))

(defn coerce-response-interceptor
  "Like reitit.http.coercion/coerce-response-interceptor, except it never coerces/validates InputStream body responses"
  []
  {:name    ::coerce-response
   :spec    ::responses
   :compile (fn [{:keys [coercion responses]} opts]
              (cond
                ;; no coercion, skip
                (not coercion) nil
                ;; just coercion, don't mount
                (not responses) {}
                ;; mount
                :else
                (if-let [coercer (coercion/response-coercer coercion responses opts)]
                  {:leave (fn [{:keys [request response] :as ctx}]
                            (->> (cond->> response
                                          (not (instance? InputStream (:body response)))
                                          (coercer request))
                                 (assoc ctx :response)))}
                  {})))})

(defn default-interceptors [muuntaja]
  [(mi/format-interceptor muuntaja)
   ::exception-interceptor
   (coerce-response-interceptor)
   (parameters/parameters-interceptor)
   (rhc/coerce-request-interceptor)])

(defn- try-parse-inst [s]
  (if (string? s)
    (try
      (Instant/parse ^String s)
      (catch Exception _
        s))
    s))

(defn -json-transformer [strip-extra-keys?]
  (mt/transformer
    {:name     :inst
     :decoders {'inst? try-parse-inst}
     :encoders {'inst? mt/-any->string}}
    (when strip-extra-keys?
      (-> (mt/strip-extra-keys-transformer)
          m/-transformer-chain
          first
          (update-vals #(dissoc % :map-of))))
    mt/default-value-transformer
    mt/json-transformer))

(def string-transformer
  (mt/transformer
    {:name     :inst
     :decoders {'inst? try-parse-inst}
     :encoders {'inst? mt/-any->string}}
    mt/string-transformer
    mt/default-value-transformer))

(def default-coerce-options
  (let [json-transformer (-json-transformer false)]
    {:transformers {:body     {:default json-transformer
                               :formats {"application/json" json-transformer}}
                    :string   {:default string-transformer}
                    :response {:default json-transformer}}
     :compile      mu/open-schema}))

(defn- attach-path-params [expand-map path-params routes-data]
  (let [expand-map (atom expand-map)
        routes-data (mapv (fn attach-path-parameters
                            ([d] (attach-path-parameters [] d))
                            ([parent-path-params original]
                             (let [path (prism/vec-first original)
                                   data (subvec original 1)
                                   first-data (prism/vec-first data)
                                   path-params (->> (ri/parse path nil)
                                                    :path-params
                                                    (into parent-path-params
                                                          (map (fn [kw] [kw (or (path-params kw) string?)]))))
                                   attach-method-path-params-fn (fn [data] (update-in
                                                                             data
                                                                             [:parameters :path]
                                                                             (fnil identity (into [:map] path-params))))]
                               (cond
                                 (vector? first-data) (into [path] (map #(attach-path-parameters path-params %)) data)
                                 (and (map? first-data)
                                      (vector? (second data))) (->> (subvec data 1)
                                                                    (into [path first-data]
                                                                          (map #(attach-path-parameters path-params %))))
                                 (seq path-params) [path
                                                    (cond
                                                      (fn? first-data) {:handler    first-data
                                                                        :parameters {:path (into [:map] path-params)}}
                                                      (:handler first-data) (attach-method-path-params-fn first-data)
                                                      (map? first-data) (update-vals
                                                                          first-data
                                                                          #(if (keyword? %)
                                                                             (do (swap!
                                                                                   expand-map
                                                                                   update
                                                                                   %
                                                                                   attach-method-path-params-fn)
                                                                                 %)
                                                                             (attach-method-path-params-fn %))))]
                                 :else original))))
                          routes-data)]
    {:expand-map  @expand-map
     :routes-data routes-data}))

(defn create-handler [routes-data & {:keys [expand expand-map coerce-options
                                            interceptors muuntaja
                                            inject-match? inject-router?
                                            attach-path-params? path-params
                                            default-options-endpoint
                                            compile base-path
                                            default-handler]
                                     :or   {muuntaja            muuntaja-instance
                                            path-params         {}
                                            attach-path-params? true
                                            inject-match?       false
                                            inject-router?      false}}]
  (let [{:keys [expand-map routes-data]} (if attach-path-params?
                                           (attach-path-params expand-map path-params routes-data)
                                           (prism/kw-map expand-map routes-data))
        expand (or expand
                   (when (seq expand-map)
                     (reference-expand expand-map))
                   r/expand)
        interceptors (or interceptors
                         (default-interceptors muuntaja))
        coercion (-> (merge default-coerce-options coerce-options)
                     mc/create)]
    (rhttp/ring-handler
      (rhttp/router
        routes-data
        (cond-> {:expand                      expand
                 :validate                    rs/validate
                 :data                        {:coercion coercion}
                 :reitit.interceptor/registry {::exception-interceptor (exception-interceptor)}}
                default-options-endpoint (assoc ::rhttp/default-options-endpoint default-options-endpoint)
                base-path (assoc :path base-path)
                compile (assoc :compile compile)))

      (or default-handler
          (ring/routes
            (ring/create-default-handler)))

      {:executor       sieppari/executor
       :interceptors   interceptors
       :inject-match?  inject-match?
       :inject-router? inject-router?})))

(deftype ChunkedFlushingInputStream [stream])
(def ^:private buffer-size 1024)

(extend-protocol rcp/StreamableResponseBody
  ChunkedFlushingInputStream
  (write-body-to-stream [^ChunkedFlushingInputStream cfis
                         _
                         ^OutputStream output-stream]
    (with-open [_ output-stream]
      (let [^InputStream stream (.-stream cfis)
            buffer (make-array Byte/TYPE buffer-size)]
        (loop [available (.available stream)]
          (let [bytes-to-read (if (pos? available)
                                (min available buffer-size)
                                buffer-size)
                remaining-available (- available bytes-to-read)
                size (.read stream buffer 0 bytes-to-read)]
            (when (pos? size)
              (.write output-stream buffer 0 size)
              (if (pos? remaining-available)
                (recur (long remaining-available))
                (do (when (= available size)
                      (.flush output-stream))
                    (recur (.available stream)))))))))))

(defn register-iteration-streaming! []
  (extend-protocol rcp/StreamableResponseBody
    Seqable
    (write-body-to-stream [it
                           _
                           ^OutputStream output-stream]
      (with-open [_ output-stream]
        (doseq [^ByteArrayOutputStream v it]
          (.writeTo v output-stream)
          (.flush output-stream))))))

(comment
  (do (require '[prism.http-server :as hs])
      (require '[prism.http :as http])
      (require '[clojure.java.io :as io]))
  (->> (create-handler
         [["/test/:abc"
           ["" {:get  ::test
                :post ::test}]
           ["/body" {:get {:handler    (fn [req]
                                         (pr req)
                                         (println)
                                         {:status 201
                                          :body   {:with "body"}})
                           :parameters {:body [:map [:a int?]]
                                        :path map?}}}]
           ["/error" {:handler    (fn [req]
                                    (pr req)
                                    (println)
                                    {:status 418})
                      :parameters {:body [:map
                                          [:jay [:enum "son" "daughter"]]]}}]
           ["/error2" (fn [req]
                        (pr req)
                        (println)
                        {:status 418})]]]
         :expand-map {::test {:handler (fn [req]
                                         (pr req)
                                         (println)
                                         {:status 200})}})
       hs/start-server!
       (def server))
  (.stop server)
  (:port (prism/config))
  (->> (http/request {:method :get
                      :url    (str "http://localhost:" (-> (prism/config) :port) "/openapi.json")})
       :body
       (spit (io/file "dev/openapi.json")))
  (http/request {:method :get
                 :url    (str "http://localhost:" (-> (prism/config) :port) "/test/value")})
  (http/request {:method           :get
                 :throw-exceptions false
                 :as               :auto
                 :json             {:a 1 :b 2}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/123/body")})
  (http/request {:method           :put
                 :timeout          100000
                 :as               :auto
                 :throw-exceptions false
                 :json             {:jay "x"}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/value/error")})

  (->> (create-handler
         [["/headers" (fn [_req] {:status  200
                                  :headers {"single"            "abc"
                                            "single_with_semis" "abc; xyz;123"
                                            "multi"             ["abc" "def" "xyz"]}})]])
       hs/start-server!
       (def header-test-server))
  (-> (http/request {:method :get
                     :url    (str "http://localhost:" (-> (prism/config) :port) "/headers")})
      :headers)
  (.stop header-test-server))
