(ns prism.services
  (:require
    [prism.core :refer [defdelayed] :as prism]
    [prism.http :as http]
    [prism.internal.classpath :as cp])
  (:import
    (java.net ConnectException)))

(defn- host-port-url [{:keys [Address ServicePort]}]
  (format "http://%s:%s" Address ServicePort))

(defdelayed ^:private consul-svc (:consul-url (prism/config)))

(defdelayed ^:private static-services (merge {:consul [(consul-svc)]}
                                             (:service-defaults (prism/config))))

(defn- lookup-svc [{:keys [service tag]}]
  (->> (http/request (cond-> {:url (str (consul-svc) "/v1/catalog/service/" (name service))
                              :as  :json}
                             tag (assoc :query-params {:filter (format "ServiceTags contains \"%s\"" (name tag))})))
       :body
       (mapv host-port-url)
       not-empty))

(defdelayed ^:private service-registry (:services (prism/config)))

(defn- lookup-svc-kw [kw]
  (if-let [svc (kw (service-registry))]
    (lookup-svc svc)
    (throw (IllegalArgumentException. (format "Service %s not found in services registry: '%s'" kw (service-registry))))))

(declare ^:private ^:redef service!)
(declare ^:private ^:redef invalidate-service!)
(cp/if-ns 'clojure.core.cache.wrapped
  (do
    (require 'clojure.core.cache)
    (defdelayed ^:private ^:redef
      services (let [ttl (:services-ttl (prism/config))
                     cache (if (pos? ttl)
                             (clojure.core.cache/ttl-cache-factory (static-services) :ttl ttl)
                             (clojure.core.cache/basic-cache-factory (static-services)))]
                 (atom cache
                       :validator (fn validate-services-urls [c]
                                    (every? some? (vals c))))))

    (defn invalidate-service! [service-kw]
      (when-not (service-kw (static-services))
        (clojure.core.cache.wrapped/evict (services) service-kw)))

    (defn- service! [service-kw]
      (or (service-kw (static-services))
          (try
            (clojure.core.cache.wrapped/lookup-or-miss (services) service-kw lookup-svc-kw)
            (catch IllegalStateException _)))))
  (do
    (defdelayed ^:private ^:redef services (atom (static-services)))

    (defn- resolve-service! [service-kw]
      (when-let [resolved (lookup-svc-kw service-kw)]
        (-> (swap! (services) assoc service-kw resolved)
            service-kw)))

    (defn- service! [service-kw]
      (or (service-kw @(services))
          (resolve-service! service-kw)))

    (defn invalidate-service! [service-kw]
      (when-not (service-kw (static-services))
        (swap! (services) dissoc service-kw)))))

(defn service-url! [service-kw]
  (-> (service! service-kw)
      rand-nth))

(defn- pathed-service-url! [service-kw path]
  (some-> (service-url! service-kw)
          (str path)))

(defn- handle-connect-exception! [service path req previous throw?]
  (invalidate-service! service)
  (if-let [url (pathed-service-url! service path)]
    (http/request (assoc req :url url))
    (cond-> (ex-info "Could not resolve domain name for service" {:service  service
                                                                  :status   503
                                                                  :previous previous})
            throw? throw)))

(defn service-request [service path {:keys [catch-exceptions?] :as req}]
  (if-let [url (pathed-service-url! service path)]
    (if catch-exceptions?
      (let [result (http/request (assoc req :url url))]
        (if (prism/ex-caused-by? result ConnectException)
          (handle-connect-exception! service path req url false)
          result))
      (try
        (http/request (assoc req :url url))
        (catch Exception ex
          (if (prism/ex-caused-by? ex ConnectException)
            (handle-connect-exception! service path req url true)
            (throw ex)))))
    (cond-> (ex-info "Could not resolve domain name for service"
                     {:service service
                      :path    path
                      :status  503})
            (not catch-exceptions?) throw)))

(comment
  (service-url! :greptime)
  (invalidate-service! :consul)
  (service-url! :consul)
  (service-url! :x)
  (-> (prism/config)
      :services
      vals
      first
      lookup-svc))
