(ns prism.services
  (:require
    [clojure.string :as str]
    [prism.core :refer [defdelayed] :as prism]
    [prism.http :as http]
    [prism.internal.classpath :as cp])
  (:import
    (java.net ConnectException)))

(defn- health-entry->url [{{:keys [Address Port]} :Service}]
  (format "http://%s:%s" Address Port))

(defdelayed ^:private consul-svc (:consul-url (prism/config)))

(defdelayed ^:private static-services (merge {:consul [(consul-svc)]}
                                             (:service-defaults (prism/config))))

(defn- lookup-svc [{:keys [service tag]}]
  (->> (http/request (cond-> {:url          (str (consul-svc) "/v1/health/service/" (name service))
                              :as           :json
                              :query-params {:passing true}}
                             tag (assoc-in [:query-params :tag] (name tag))))
       :body
       (mapv health-entry->url)
       not-empty))

(defdelayed ^:private service-registry (:services (prism/config)))

(defn- lookup-svc-kw [kw]
  (if-let [svc (kw (service-registry))]
    (lookup-svc svc)
    (throw (IllegalArgumentException. (format "Service %s not found in services registry: '%s'" kw (service-registry))))))

(declare ^:private ^:redef service!)
(declare ^:private ^:redef invalidate-service!)
(cp/if-ns 'clojure.core.cache.wrapped
  (do
    (require 'clojure.core.cache)
    (defdelayed ^:private ^:redef
      services (let [ttl (:services-ttl (prism/config))
                     cache (if (pos? ttl)
                             (clojure.core.cache/ttl-cache-factory (static-services) :ttl ttl)
                             (clojure.core.cache/basic-cache-factory (static-services)))]
                 (atom cache
                       :validator (fn validate-services-urls [c]
                                    (every? some? (vals c))))))

    (defn invalidate-service! [service-kw]
      (when-not (service-kw (static-services))
        (clojure.core.cache.wrapped/evict (services) service-kw)))

    (defn- service! [service-kw]
      (or (service-kw (static-services))
          (try
            (clojure.core.cache.wrapped/lookup-or-miss (services) service-kw lookup-svc-kw)
            (catch IllegalStateException _)))))
  (do
    (defdelayed ^:private ^:redef services (atom (static-services)))

    (defn- resolve-service! [service-kw]
      (when-let [resolved (lookup-svc-kw service-kw)]
        (-> (swap! (services) assoc service-kw resolved)
            service-kw)))

    (defn- service! [service-kw]
      (or (service-kw @(services))
          (resolve-service! service-kw)))

    (defn invalidate-service! [service-kw]
      (when-not (service-kw (static-services))
        (swap! (services) dissoc service-kw)))))

(defn service-url! [service-kw]
  (-> (service! service-kw)
      rand-nth))

(defn- retry-service-url! [service-kw previous-url]
  (let [urls (service! service-kw)]
    (case (count urls)
      0 nil
      1 (prism/vec-first urls)
      (-> (filterv #(not (str/starts-with? previous-url %)) urls)
          rand-nth))))

(defn- handle-connect-exception! [service path req previous throw?]
  (invalidate-service! service)
  (if-let [url (some-> (retry-service-url! service previous)
                       (str path))]
    (http/request (assoc req :url url))
    (cond-> (ex-info "Could not resolve domain name for service" {:service  service
                                                                  :status   503
                                                                  :previous previous})
            throw? throw)))

(defn- connection-exception? [ex]
  (prism/ex-caused-by? ex ConnectException))

(defn service-request [service path {:keys [catch-exceptions?] :as req}]
  (if-let [url (some-> (service-url! service)
                       (str path))]
    (if catch-exceptions?
      (let [result (http/request (assoc req :url url))]
        (if (connection-exception? result)
          (handle-connect-exception! service path req url false)
          result))
      (try
        (http/request (assoc req :url url))
        (catch Exception ex
          (if (connection-exception? ex)
            (handle-connect-exception! service path req url true)
            (throw ex)))))
    (cond-> (ex-info "Could not resolve domain name for service"
                     {:service service
                      :path    path
                      :status  503})
            (not catch-exceptions?) throw)))

(comment
  (lookup-svc {:service :greptime-0-13})
  (invalidate-service! :consul)
  (service-url! :consul)
  (service-url! :x)
  (-> (prism/config)
      :services
      vals
      first
      lookup-svc))
