(ns prism.aws
  (:require
    [prism.http :as http]
    [prism.services :as services]))

(defn- prepare-aws-request [req]
  (update req :query-params
          (fn [qp]
            (when qp
              (into (sorted-map) qp)))))

(defprotocol AwsClient
  (request [_this path request]))

(defrecord ServiceClient [service-kw aws-config]
  AwsClient
  (request [_ path request]
    (->> (assoc request :aws aws-config)
         prepare-aws-request
         (services/service-request service-kw path))))

(defrecord StaticUrlClient [url aws-config]
  AwsClient
  (request [_ path request]
    (->> (assoc request :aws aws-config
                        :url (str url path))
         prepare-aws-request
         http/request)))

(defrecord DynamicUrlClient [->url aws-config]
  AwsClient
  (request [_ path request]
    (->> (assoc request :aws aws-config
                        :url (str (->url aws-config request) path))
         prepare-aws-request
         http/request)))

(defn- default-url [{:keys [service region]}]
  (str "https://"
       service
       (when region \.)
       region
       ".amazonaws.com"))

(defn create-client [type client-config]
  (let [aws-config (-> (select-keys client-config [:region :service])
                       (assoc :credentials {:aws/access-key-id     (:access-key client-config)
                                            :aws/secret-access-key (:secret-key client-config)}))]
    (case type
      ::default (->StaticUrlClient (default-url client-config) aws-config)
      ::static-url (->StaticUrlClient (:url client-config) aws-config)
      ::service (->ServiceClient (:consul-service client-config) aws-config)
      ::dynamic (->DynamicUrlClient (:url-fn client-config) aws-config))))
