(ns prism.redis
  (:require
    [prism.core :as prism :refer [defdelayed]]
    [taoensso.carmine :as car]
    [taoensso.trove :as trove]))

(defn- connection-pool [cfg]
  {:pool (car/connection-pool {})
   :spec cfg})

(defdelayed conns (-> (prism/config)
                      :redis
                      (update-vals connection-pool)))

(defmacro wcar* [conn-name & body]
  `(let [c# (~conn-name (conns))]
     (try
       (car/wcar c# ~@body)
       (catch Exception e#
         (trove/log! {:level :error
                      :error e#})))))

(defn hexpire-at [k expire-at-secs members]
  (car/redis-call (into ["hexpireat" k expire-at-secs "FIELDS" (str (count members))]
                        members)))

(defn hset [k score-members]
  (when (seq score-members)
    (apply car/hset k score-members)))

(defn lpush [k members]
  (when (seq members)
    (apply car/lpush k members)))

(defn set-kv!* [c {:keys [k v expire expire-at get? exclusive]}]
  (when (some? v)
    (let [resp (wcar*
                 c
                 (apply car/set
                        (cond-> [k v]
                                (#{:nx :xx} exclusive) (conj exclusive)
                                get? (conj :get)
                                (integer? expire) (conj :ex expire)
                                (integer? expire-at) (conj :exat expire-at))))]
      (if get?
        resp
        v))))

(defn get-val* [c k]
  (wcar* c (car/get k)))

(defn get-or-set!* [c {:keys [k f] :as entry}]
  (or (get-val* c k)
      (some->> (f)
               (assoc entry :v)
               (set-kv!* c))))

(defn hset!* [c {:keys [k member v]}]
  (when v
    (wcar* c (car/hset k member v))
    v))

(defn hdel!* [c {:keys [k member]}]
  (wcar* c (car/hdel k member)))

(defn del!* [c k]
  (wcar* c (car/del k)))

(defn hvals* [c k]
  (wcar* c (car/hvals k)))

(defn lrange-all* [c k]
  (wcar* c (car/lrange k 0 -1)))

(comment
  (get-val* :local :test2)
  (wcar* :local (car/get :test))
  (del!* :local :test2)
  (get-or-set!*
    :local
    {:k :test2
     :f (fn [] "abc")}))
