(ns prism.apache-http-client
  (:require
    [clojure.string :as str]
    [prism.core :refer [defdelayed]]
    [prism.http-client-protocol :as hcp])
  (:import
    (java.io Closeable File InputStream)
    (java.net URI)
    (java.util.concurrent CompletableFuture ExecutorService Executors TimeUnit)
    (java.util.function Function)
    (org.apache.hc.client5.http.impl.async CloseableHttpAsyncClient HttpAsyncClients)
    (org.apache.hc.core5.concurrent FutureCallback)
    (org.apache.hc.core5.http ClassicHttpRequest Header HttpResponse HttpVersion Message MessageHeaders)
    (org.apache.hc.core5.http.io.entity EmptyInputStream)
    (org.apache.hc.core5.http.io.support ClassicRequestBuilder)
    (org.apache.hc.core5.http.nio AsyncEntityConsumer AsyncEntityProducer)
    (org.apache.hc.core5.http.nio.entity BasicAsyncEntityProducer FileEntityProducer StringAsyncEntityProducer)
    (org.apache.hc.core5.http.nio.support BasicRequestProducer BasicResponseConsumer)
    (org.apache.hc.core5.http.nio.support.classic AbstractClassicEntityConsumer)
    (prism.internal InputStreamEntityProducer)))

(defn- add-headers! ^ClassicRequestBuilder [r headers]
  (reduce-kv ^[String String] ClassicRequestBuilder/.addHeader r headers))

(defn- req-uri "As per hato.client/ring-request->HttpRequest"
  ^URI [{:keys [scheme server-name server-port uri query-string]}]
  (-> (str (name scheme)
           "://"
           server-name
           (some->> server-port (str ":"))
           uri
           (some->> query-string (str "?")))
      URI.))

(defn- ->http-version ^HttpVersion [kw]
  (case kw
    :http-1.1 HttpVersion/HTTP_1_1
    :http2 HttpVersion/HTTP_2_0
    nil))

(defn- http-version->kw [^HttpVersion http-version]
  (case (.getMajor http-version)
    3 :http3
    2 :http2
    1 (keyword (str "http-1." (.getMinor http-version)))))

(defn- ->headers-map [^MessageHeaders mh]
  (as-> (.headerIterator mh) headers
        (iterator-seq headers)
        (reduce
          (fn attach-header! [acc ^Header h]
            (let [header-name (.getName h)]
              (update acc header-name
                      (fn add-header [existing]
                        (let [v (.getValue h)]
                          (cond
                            (nil? existing) v
                            (string? existing) [existing v]
                            (vector? existing) (conj existing v)))))))
          {}
          headers)
        (update-keys headers str/lower-case)))

(defn- response-map [^Message response
                     {:keys [request-map
                             ^ClassicHttpRequest request
                             http-client]}]
  (let [response-head ^HttpResponse (.getHead response)]
    {:uri         (.getUri request)
     :status      (.getCode response-head)
     :body        (or (.getBody response)
                      EmptyInputStream/INSTANCE)
     :headers     (->headers-map response-head)
     :version     (-> (.getVersion response-head)
                      http-version->kw)
     :http-client (dissoc http-client :executor-service :client-impl) ;excluded as these are not serialisable by jackson
     :request     (assoc request-map :http-request request)}))

(defn- entity-consumer ^AsyncEntityConsumer [stream-buffer-size executor-service]
  (proxy [AbstractClassicEntityConsumer] [stream-buffer-size executor-service]
    (consumeData [_content-type input-stream]
      input-stream)))

(defn- entity-producer ^AsyncEntityProducer [body stream-buffer-size executor-service]
  (cond
    (nil? body) nil
    (instance? InputStream body) (InputStreamEntityProducer. body stream-buffer-size nil executor-service)
    (string? body) (StringAsyncEntityProducer. ^String body nil)
    (bytes? body) (BasicAsyncEntityProducer. ^bytes body nil)
    (instance? File body) (FileEntityProducer. ^File body nil)
    :else (throw (IllegalArgumentException. (str "Body not supported: " (class body))))))

(defn- completable-future-callback [^CompletableFuture cf request-details]
  (reify FutureCallback
    (completed [_ v]
      (->> (response-map v request-details)
           (.complete cf)))
    (failed [_ ex]
      (.completeExceptionally cf ex))
    (cancelled [_]
      (.cancel cf true))))

(defrecord ApacheHttpClient [^CloseableHttpAsyncClient client-impl
                             ^ExecutorService executor-service
                             stream-buffer-size]
  hcp/HttpClient
  (hcp/client-type [_] :apache)
  (hcp/request* [this {:keys [request-method version headers body timeout async?] :as req} respond raise]
    (let [version (->http-version version)
          method (or (some-> request-method name str/upper-case)
                     "GET")
          request (-> (ClassicRequestBuilder/create method)
                      (.setUri (req-uri req))
                      (.setVersion version)
                      (add-headers! headers)
                      .build)
          request-details {:request-map req
                           :request     request
                           :http-client this}
          producer (BasicRequestProducer. request (entity-producer body stream-buffer-size executor-service))
          consumer (BasicResponseConsumer. (entity-consumer stream-buffer-size executor-service))]
      (if-not async?
        (-> (.execute client-impl producer consumer nil nil nil)
            (.get timeout TimeUnit/MILLISECONDS)
            (response-map request-details))
        (let [callback-cf (CompletableFuture.)
              cf (-> (.thenApply callback-cf ^Function respond)
                     (.exceptionally ^Function raise)
                     (.orTimeout ^long timeout TimeUnit/MILLISECONDS))]
          (->> (completable-future-callback callback-cf request-details)
               (.execute client-impl producer consumer nil nil))
          cf))))
  Closeable
  (^void close [_]
    (.close client-impl)
    (.close executor-service)))

(defdelayed default-client-impl (-> (HttpAsyncClients/custom)
                                    .disableRedirectHandling
                                    .disableCookieManagement
                                    .build
                                    (doto (.start))))

(defdelayed default-http-client
  (->ApacheHttpClient
    (default-client-impl)
    (Executors/newVirtualThreadPerTaskExecutor)
    4096))

(comment
  (prism.json/write-json-string
    (dissoc (default-http-client) :executor-service))
  (->http-version :http-1.1)
  (hcp/client-type (default-http-client))
  (-> (hcp/request*
        (default-http-client)
        {:scheme         "http"
         :server-name    "localhost"
         :server-port    8084
         :timeout        10000
         :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
         :request-method :post
         ;:version        :http2
         :async?         true
         :headers        {"test" "header"}
         :body           "my body"}
        (fn [x] (println x) x)
        #(throw %))
      .get
      (update :body slurp))
  (hcp/request*
    (default-http-client)
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    8084
     :timeout        10000
     :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
     :request-method :post
     :headers        {"test" "header"}
     :body           "my body"}
    nil nil)
  (hcp/request*
    (default-http-client)
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    2226
     :timeout        10000
     :uri            "/tenants/961478ea-4c14-4830-8f50-b504965adfe0/alerts"
     :request-method :get
     :version        :http-1.1}
    nil nil)
  (hcp/request*
    (default-http-client)
    {:scheme         "https"
     :server-name    "gitlab.com"
     ;:server-port    2999
     ;:uri            "/test/abc/error2"
     :request-method :get
     :timeout        10000
     ;:headers        {"test" "header"}
     :body           "my body"}
    nil nil))
