(ns prism.json
  (:require
    [clojure.stacktrace :as stacktrace]
    [clojure.string :as str]
    [jsonista.core :as j]
    [prism.core :as prism]
    [prism.internal.classpath :as cp])
  (:import
    (com.fasterxml.jackson.databind ObjectMapper)
    (java.io Writer)))

(def object-mapper (j/object-mapper
                     {:mapper                     (.copy ^ObjectMapper j/keyword-keys-object-mapper)
                      :do-not-fail-on-empty-beans true}))

(defn json->clj [o]
  (j/read-value o object-mapper))

(defn write-json-string [o]
  (j/write-value-as-string o object-mapper))

(defn json->clj-checked [s]
  (when-not (str/blank? s)
    (json->clj s)))

(def configure-logging! (cp/missing-dep "timbre" "aero"))
(cp/when-ns 'taoensso.timbre
  (cp/when-ns 'aero.core
    (let [structure-log (fn structure-log [{:keys [msg_ ?err ?ns-str ?line level timestamp_ context]}]
                          (let [thread (Thread/currentThread)
                                thread-str (or (not-empty (.getName thread))
                                               (str "virtual-thread-" (.threadId thread)))]
                            (cond-> (assoc context
                                      :caller (str ?ns-str \: ?line)
                                      :level (name level)
                                      :thread thread-str
                                      :msg (force msg_)
                                      :ts (force timestamp_))
                                    ?err (assoc :stacktrace (with-out-str
                                                              (stacktrace/print-cause-trace ?err)))
                                    (seq (ex-data ?err)) (assoc :ex-data (write-json-string (ex-data ?err))))))]
      (defn configure-logging! []
        (taoensso.timbre/set-config!
          {:min-level      (-> (prism/config) :log-level keyword)
           :ns-filter      #{"*"}
           :middleware     []
           :timestamp-opts taoensso.timbre/default-timestamp-opts
           :output-fn      structure-log
           :appenders      {:println {:enabled?   true
                                      :async?     false
                                      :min-level  nil
                                      :rate-limit nil
                                      :output-fn  :inherit
                                      :fn         (fn json-appender-fn [{:keys [output_]}]
                                                    (.write ^Writer *err*
                                                            (str
                                                              (write-json-string (force output_))
                                                              taoensso.encore/system-newline))
                                                    (.flush ^Writer *err*))}}})))))

(cp/when-ns 'muuntaja.format.json
  (def muuntaja-format
    {:name    "application/json"
     :decoder [muuntaja.format.json/decoder {:mapper object-mapper}]
     :encoder [muuntaja.format.json/encoder {:mapper object-mapper}]}))

(comment
  (configure-logging!)
  (taoensso.timbre/info "test"))
