(ns prism.postgres
  (:require
    [clojure.core.cache.wrapped :as ccw]
    [clojure.string :as str]
    [honey.sql :as hsql]
    [next.jdbc :as jdbc]
    [next.jdbc.connection :as connection]
    [next.jdbc.date-time :as date-time]
    [next.jdbc.prepare :refer [SettableParameter]]
    [next.jdbc.result-set :refer [ReadableColumn as-unqualified-kebab-maps]]
    [prism.core :refer [defdelayed] :as prism]
    [prism.internal.classpath :as cp])
  (:import
    (clojure.lang IObj IPersistentMap IPersistentVector)
    (com.zaxxer.hikari HikariDataSource)
    (java.sql Connection PreparedStatement Timestamp Types)
    (java.time Instant)
    (java.util Date UUID)
    (org.postgresql.util PGobject)))

(cp/when-ns 'prism.json
  (defn- ->pgobject
    "Transforms Clojure data to a PGobject that contains the data as
    JSON. PGObject type defaults to `jsonb` but can be changed via
    metadata key `:pgtype`"
    [x]
    (let [pg-type (or (:pg-type (meta x)) "jsonb")]
      (doto (PGobject.)
        (.setType pg-type)
        (.setValue (prism.json/write-json-string x)))))

  (defn- <-pgobject
    "Transform PGobject containing `json` or `jsonb` value to Clojure
    data."
    [^PGobject v]
    (let [type (.getType v)
          value (.getValue v)]
      (if (#{"jsonb" "json"} type)
        (let [v (prism.json/json->clj value)]
          (cond-> v
                  (instance? IObj v) (with-meta {:pg-type type})))
        value)))

  (extend-protocol SettableParameter
    IPersistentMap
    (set-parameter [m ^PreparedStatement s i]
      (.setObject s i (->pgobject m)))

    IPersistentVector
    (set-parameter [v ^PreparedStatement s i]
      (.setObject s i (->pgobject v))))

  (extend-protocol ReadableColumn
    PGobject
    (read-column-by-label [^PGobject v _]
      (<-pgobject v))
    (read-column-by-index [^PGobject v _ _]
      (<-pgobject v))))

(extend-protocol SettableParameter
  Date
  (set-parameter [v ^PreparedStatement s i]
    (.setTimestamp s i (-> v
                           (inst-ms)
                           (Timestamp.))))

  Instant
  (set-parameter [v ^PreparedStatement s i]
    (.setTimestamp s i (Timestamp/from v)))

  String
  (set-parameter [v ^PreparedStatement s ^long i]
    (.setObject s i v Types/OTHER)))

(extend-protocol ReadableColumn
  UUID
  (read-column-by-label [^PGobject v _]
    (str v))
  (read-column-by-index [^PGobject v _ _]
    (str v)))

(defn- json-get-str [_ [& exprs]]
  (let [[sqls params] (hsql/format-expr-list exprs)
        sql (-> (repeat (- (count sqls) 2) " -> ")
                vec
                (into [" ->> " nil])
                (->> (interleave sqls))
                str/join)]
    (into [sql] params)))

(date-time/read-as-instant)
(hsql/register-fn! :->> json-get-str)
(hsql/register-op! :-> :variadic true)
(hsql/register-op! :||)

(defn kw->param [kw]
  (keyword (str \? (name kw))))

(defmacro ->params [& kws]
  (into {} (map (fn [kw] [kw (kw->param kw)])) kws))

(defdelayed ^HikariDataSource data-src
  (let [{:keys [pool]
         :as   postgres-config} (-> (prism/config)
                                    :postgres)]
    (->> (assoc postgres-config :dbtype "postgres")
         connection/jdbc-url
         (assoc pool :jdbcUrl)
         (connection/->pool HikariDataSource))))

(defdelayed ^:private query-cache (ccw/soft-cache-factory {}))

(declare ^:dynamic ^Connection *c*)
(declare ^:dynamic ^Connection *txn*)

(defn connect ^Connection [] (jdbc/get-connection (data-src)))

(defmacro with-conn [& body]
  `(with-open [c# (connect)]
     (binding [*c* c#]
       ~@body)))

(defmacro with-txn [& body]
  `(jdbc/with-transaction
     [~'txn (data-src)]
     (binding [*txn* ~'txn
               *c* ~'txn]
       ~@body)))

(defn- format-sql [statement {cached? ::cached?
                              :as     options
                              :or     {cached? true}}]
  (let [params (dissoc options ::cached?)
        opts (cond-> {}
                     (seq params) (assoc :params params)
                     cached? (assoc :cache (query-cache)))]
    (hsql/format statement opts)))

(defn execute! [statement {:keys [::return-keys] :as opts}]
  (jdbc/execute! *c*
                 (format-sql statement opts)
                 (cond-> {:builder-fn as-unqualified-kebab-maps}
                         (seq return-keys) (assoc :return-keys (mapv (comp prism/vec-first hsql/format-expr) return-keys)))))

(defn batch-insert! [table params]
  (when (seq params)
    (let [params-template (-> (first params)
                              (update-vals (constantly 0)))
          ->ordered-params (apply juxt (keys params-template))
          statement (-> (format-sql {:insert-into table
                                     :values      [params-template]}
                                    {::cached? false})
                        (subvec 0 1))]
      (with-open [ps (jdbc/prepare *c* statement)]
        (->> (mapv ->ordered-params params)
             (jdbc/execute-batch! ps))))))

(defn exists? [exists-query params]
  (-> (execute! {:select [[true :exists]]
                 :where  [:exists (assoc exists-query :select [1])]}
                params)
      first
      :exists))

(defn updated? [result]
  (-> result prism/vec-first :next.jdbc/update-count (> 0)))

(defn rollback-txn! []
  (.rollback *txn*))
