(ns prism.greptime
  (:require
    [clojure.string :as str]
    [honey.sql :as hsql]
    [honey.sql.protocols :as hsp]
    [prism.core :refer [defdelayed] :as prism]
    [prism.ilp :as ilp]
    [prism.internal.classpath :as cp]
    [prism.json :as json]
    [prism.services :as services])
  (:import
    (clojure.lang IPersistentList IPersistentSet IPersistentVector ISeq Sequential)
    (java.time Instant)
    (java.util.concurrent TimeUnit)))

(defn- format-with [_ withs]
  (let [{:keys [clause params]} (reduce-kv
                                  (fn [acc k v]
                                    (let [[sql param] (hsql/format-expr v)
                                          k (hsql/format-entity k)]
                                      (-> (update acc :clause str k "=" sql ",")
                                          (cond-> (some? param) (update :params conj param)))))
                                  {:clause "with("
                                   :params []}
                                  withs)
        clause (-> (subs clause 0 (dec (count clause)))
                   (str ")"))]
    (into [clause] params)))

(defn- format-engine [_ engine]
  [(str "engine=" engine)])

(defn- sqlize-coll [coll]
  (as-> (mapv hsp/sqlize coll) $
        (str/join ", " $)
        (str \( $ \))))

(extend-protocol hsp/InlineValue
  Sequential
  (sqlize [this] (sqlize-coll this))
  IPersistentVector
  (sqlize [this] (sqlize-coll this))
  IPersistentList
  (sqlize [this] (sqlize-coll this))
  IPersistentSet
  (sqlize [this] (sqlize-coll this))
  ISeq
  (sqlize [this] (sqlize-coll this)))

(defn format-interval [_ interval]
  [(str "INTERVAL(" (hsql/format-entity interval) ")")])

(defn format-set-ttl [_ ttl]
  (let [[s & p] (hsql/format-expr ttl)]
    (into [(str "SET 'ttl'=" s)] p)))

(hsql/register-clause! :insert-into :insert-into :select)
(hsql/register-clause! :with-metadata format-with nil)
(hsql/register-clause! :engine format-engine :with-metadata)
(hsql/register-clause! :tags :values :join-by)
(hsql/register-clause! :interval format-interval :order-by)
(hsql/register-clause! :set-ttl format-set-ttl :add-column)
(hsql/register-clause! :as-select :select :create-table)
(hsql/register-clause! :sink-to :insert-into :as-select)
(hsql/register-clause! :create-flow :insert-into :sink-to)

(comment
  (hsql/format
    {:alter-table :x :set-ttl "1d"}
    {:inline true})
  (hsql/format
    {:alter-table :x :set-ttl :?ttl}
    {:inline true
     :params {:ttl "1d"}})
  (hsql/format
    {:create-table :my-table
     :with-columns [[:ts [:timestamp 9]]
                    [:tag-one :string]
                    [:tag-two :string]
                    [[:time-index :ts]]
                    [[:primary-key :tag-one :tag-two]]]})
  (hsql/format
    {:create-flow :new-flow
     :sink-to     :my-table
     :as-select   :a
     :from        :other-table
     :where       [:= :a 123]
     :group-by    [:b]}
    {}))

(defdelayed default-db (-> (prism/config) :greptime :dbname))

(defn- query-params [db] {:db db})

(defn- handle-http-result [response]
  (if-let [data (and (instance? Exception response)
                     (ex-data response))]
    (let [{:keys [body]
           :as   data} (update data :body slurp)]
      (when (re-seq #"(?i)(:?read only|follower)" body)
        (services/invalidate-service! :greptime))
      (ex-info (ex-message response) data response))
    response))

(defn- greptime-sql-request! [^String sql db]
  (-> (services/service-request
        :greptime "/v1/sql"
        {:method            :post
         :query-params      (query-params db)
         :form-params       {:sql sql}
         :catch-exceptions? true
         :retry             true
         :as                :json})
      handle-http-result))

(defn- nanos->inst ^Instant [ns]
  (Instant/ofEpochSecond 0 ns))

(defn- build-format-fn* [column-schemas]
  (let [columns (mapv
                  (fn [{:keys [name data_type]}]
                    {:name   (keyword (prism/snake->kebab name))
                     :parser (case data_type
                               "TimestampMillisecond" Instant/ofEpochMilli
                               "TimestampNanosecond" nanos->inst
                               identity)})
                  column-schemas)]
    (fn format-row [row]
      (->> (mapv vector row columns)
           (reduce (fn [r [row-column metadata]]
                     (assoc! r
                             (:name metadata)
                             ((:parser metadata) row-column)))
                   (transient {}))
           persistent!))))

(def ^:private build-format-fn (cp/if-ns 'clojure.core.memoize
                                 (clojure.core.memoize/lru build-format-fn* :lru/threshold 20)
                                 (memoize build-format-fn*)))

(defn- format-records [{:keys [schema rows]}]
  (let [format-fn (-> (:column_schemas schema)
                      build-format-fn)]
    (mapv format-fn rows)))

(comment (format-records
           {:schema {:column_schemas [{:name "field_name", :data_type "String"}
                                      {:name "greptime_value", :data_type "Float64"}
                                      {:name "greptime_timestamp", :data_type "TimestampMillisecond"}]}
            :rows   [["apps" 0.01882346944538697 1706138100000]]}))

(defn- format-success-response [{:keys [output]}]
  (mapv (fn [{:keys [records] :as res}]
          (if (nil? records)
            res
            (format-records records)))
        output))

(defn- handle-sql-response [{:keys [code] :as response} cause]
  (case (long (or code 0))
    0 (format-success-response response)
    4001 nil ;table doesn't exist - https://github.com/GreptimeTeam/greptimedb/blob/3029b47a89ca55c422a268d2d0e2ba38fbff9e95/src/common/error/src/status_code.rs
    (throw (ex-info "Error executing Greptime query" response cause))))

(defn- try-read-json [body]
  (prism/try-or (json/json->clj body) nil))

(defn- execute-sql! [sql db]
  (let [response (greptime-sql-request! sql db)]
    (if-not (instance? Exception response)
      (-> response
          :body
          (handle-sql-response nil))
      (if-let [body (-> response
                        ex-data
                        :body
                        try-read-json)]
        (handle-sql-response body response)
        (throw response)))))

(defn execute!
  ([query params] (execute! query params (default-db)))
  ([query params db]
   (-> (hsql/format query {:inline true
                           :params params})
       prism/vec-first
       (execute-sql! db))))

(defn execute-batch!
  ([queries] (execute-batch! queries (default-db)))
  ([queries db]
   (-> (eduction (map (fn [[query params]]
                        (-> (hsql/format query {:inline true
                                                :params params})
                            prism/vec-first)))
                 queries)
       (->> (str/join ";"))
       (execute-sql! db))))

(defn insert!
  ([points] (insert! points (default-db)))
  ([points db]
   (when-let [body (ilp/points->string points)]
     (-> (services/service-request
           :greptime "/v1/influxdb/write"
           {:method            :post
            :content-type      :application/x-www-form-urlencoded
            :query-params      (assoc (query-params db) "precision" "ns")
            :catch-exceptions? true
            :retry             true
            :as                :stream
            :body              body})
         handle-http-result))))

(defn- nanos->secs [ts]
  (.toSeconds TimeUnit/NANOSECONDS ts))

(defn- make-value-extractor [column->index ts-column value-column logs?]
  (let [ts-idx (get column->index ts-column)
        ts-fn (if logs? str nanos->secs)
        value-idx (get column->index value-column)]
    (fn row->value [row]
      [(ts-fn (nth row ts-idx))
       (str (nth row value-idx))])))

(defn- get-row-labels [row label-columns column->index]
  (into {}
        (map (fn [label]
               [label (nth row (get column->index label))]))
        label-columns))

(defn- format-success-as-loki [response-body {:keys [range? logs? ts-column value-column]
                                              :or   {range?       true
                                                     logs?        false
                                                     ts-column    "ts"
                                                     value-column "value"}}]
  (let [{:keys [rows] :as records} (get-in response-body [:output 0 :records])
        column-schemas (get-in records [:schema :column_schemas])
        column->index (->> (map-indexed (fn [idx schema] [(:name schema) idx]) column-schemas)
                           (into {}))
        label-columns (->> (mapv :name column-schemas)
                           (remove #{ts-column value-column}))
        extract-value (make-value-extractor column->index ts-column value-column logs?)
        result-type (if logs?
                      "streams"
                      (if range?
                        "matrix"
                        "vector"))
        labels-key (if (= result-type "streams") :stream :metric)
        values-key (if (= result-type "vector") :value :values)
        values-fn (if (= result-type "vector") first identity)]
    {:status "success"
     :data   {:resultType result-type
              :result     (->> (group-by #(get-row-labels % label-columns column->index) rows)
                               (mapv (fn [[labels group]]
                                       {labels-key labels
                                        values-key (-> (mapv extract-value group)
                                                       values-fn)})))}}))

(defn format-as-loki [response-body & {:as opts}]
  (if (-> (or (:code response-body) 0)
          long
          zero?)
    (format-success-as-loki response-body opts)
    {:status "failure"
     :data   {:error    (:error response-body)
              :type     (:code response-body)
              :result   nil
              :warnings nil}}))

(comment
  (insert! [{:measurement :new_table
             :time        (Instant/now)
             :tags        {:my-tag (str (random-uuid))}
             :fields      {:my-field "field-content-here"}}])
  (execute! [:raw "select * from new_table"]
            {})

  (execute! {:select :* :from :table-does-not-exist}
            {})
  (execute! [:raw "select * from not_a_table"]
            {}))
