(ns prism.s3
  (:require
    [clojure.data.xml :as xml]
    [clojure.java.io :as io]
    [hato.middleware :as hmw]
    [prism.aws :as aws]
    [prism.core :refer [defdelayed] :as prism])
  (:import
    (java.nio.charset StandardCharsets)
    (java.security MessageDigest)
    (java.util Base64)))

(defdelayed ^:private s3-client
  (let [{{:keys [username password region]
          :or   {region "us-east-1"}} :s3} (prism/config)]
    (aws/create-client
      ::aws/service
      {:access-key     username
       :secret-key     password
       :region         region
       :service        "s3"
       :consul-service :s3})))

(defn- s3-path
  ([bucket] (str "/" bucket))
  ([bucket key] (str (s3-path bucket) "/" key)))

(defn list-objects [{:keys [bucket delimiter prefix]
                     :or   {delimiter "/"}}
                    & {:as http-config}]
  (aws/request
    (s3-client)
    (s3-path bucket)
    (assoc
      http-config
      :method :get
      :query-params {"list-type" "2"
                     "delimiter" delimiter
                     "prefix"    prefix}
      :as :xml)))

(defn delete-object! [{:keys [bucket key]}
                      & {:as http-config}]
  (aws/request
    (s3-client)
    (s3-path bucket key)
    (assoc http-config :method :delete)))

(defn put-object! [{:keys [bucket key body]}
                   & {:as http-config}]
  (aws/request
    (s3-client)
    (s3-path bucket key)
    (assoc http-config
      :method :put
      :body body)))

(defn get-object [{:keys [bucket key]}
                  & {:as http-config}]
  (aws/request
    (s3-client)
    (s3-path bucket key)
    (-> (update http-config :as (fnil identity :stream))
        (assoc :method :get))))

(defn get-object-body [req & {:as http-config}]
  (let [{:keys [status body]} (get-object req (assoc http-config :throw-exceptions false))]
    (cond
      (hmw/unexceptional-status? status) body
      (= 404 status) nil
      :else (throw (ex-info "Failed to fetch object" (assoc req :status status :body (slurp body)))))))

(defn get-object-metadata [{:keys [bucket key]}
                           & {:as http-config}]
  (aws/request
    (s3-client)
    (s3-path bucket key)
    (assoc http-config
      :method :head
      :as :stream)))

(defn- delete-objects-request ^String [ks quiet?]
  (let [base (cond-> [:Delete {:xmlns "http://s3.amazonaws.com/doc/2006-03-01/"}]
                     quiet? (conj [:Quiet "true"]))]
    (->> (into base
               (map (fn [k] [:Object [:Key k]]))
               ks)
         xml/sexp-as-element
         xml/emit-str)))

(defn- aws-md5-header [body-bytes]
  (let [md (MessageDigest/getInstance "MD5")]
    (.update md ^bytes body-bytes)
    (-> (.encode (Base64/getEncoder) (.digest md))
        (String. StandardCharsets/US_ASCII))))

(defn delete-objects! [{:keys [bucket quiet?] :as opts}
                       & {:as http-config}]
  (let [{:keys [status]
         :as   list-response} (list-objects opts http-config)]
    (if (hmw/unexceptional-status? status)
      (when-let [keys (as-> (-> list-response :body :ListBucketResult :Contents) $
                            (if (map? $) (vector $) $)
                            (mapv :Key $)
                            (seq $))]
        (let [body (-> (delete-objects-request keys quiet?)
                       .getBytes)]
          (aws/request
            (s3-client)
            (str (s3-path bucket) "/?delete")
            (-> (assoc http-config :method :post
                                   :as :xml
                                   :body body)
                (assoc-in [:headers "Content-MD5"] (aws-md5-header body))))))
      list-response)))

(comment
  (put-object! {:bucket "test"
                :key    "some-other-key"
                :body   (io/input-stream (io/file "deps.edn"))})
  (put-object! {:bucket "test"
                :key    "some-key"
                :body   (io/input-stream (io/file "deps.edn"))})
  (list-objects {:prefix "some-"
                 :bucket "test"})
  (get-object-metadata {:bucket "test"
                        :key    "some-key"})
  (get-object {:bucket "test"
               :key    "response_test.txt"})

  (delete-objects! {:prefix "some-"
                    :quiet? true
                    :bucket "test"}))
