(ns prism.apache-http-client
  (:require
    [clojure.string :as str]
    [prism.core :refer [defdelayed]]
    [prism.http-client-protocol :as hcp])
  (:import
    (java.io Closeable InputStream)
    (java.net URI)
    (java.nio ByteBuffer)
    (java.util.concurrent CompletableFuture ExecutorService Executors TimeUnit)
    (java.util.function Function)
    (org.apache.hc.client5.http.impl.async CloseableHttpAsyncClient HttpAsyncClients)
    (org.apache.hc.core5.concurrent FutureCallback)
    (org.apache.hc.core5.http ClassicHttpRequest Header HttpResponse HttpVersion Message)
    (org.apache.hc.core5.http.io.support ClassicRequestBuilder)
    (org.apache.hc.core5.http.nio AsyncEntityConsumer AsyncEntityProducer DataStreamChannel)
    (org.apache.hc.core5.http.nio.entity BasicAsyncEntityProducer StringAsyncEntityProducer)
    (org.apache.hc.core5.http.nio.support BasicRequestProducer BasicResponseConsumer)
    (org.apache.hc.core5.http.nio.support.classic AbstractClassicEntityConsumer)))

(set! *warn-on-reflection* true)

(defn- add-headers! ^ClassicRequestBuilder [r headers]
  (reduce-kv ^[String String] ClassicRequestBuilder/.addHeader r headers))

(defn- req-uri "As per hato.client/ring-request->HttpRequest"
  ^URI [{:keys [scheme server-name server-port uri query-string]}]
  (-> (str (name scheme)
           "://"
           server-name
           (some->> server-port (str ":"))
           uri
           (some->> query-string (str "?")))
      URI.))

(defn- ->http-version ^HttpVersion [kw]
  (case kw
    :http-1.1 HttpVersion/HTTP_1_1
    :http2 HttpVersion/HTTP_2_0
    nil))

(defn- http-version->kw [^HttpVersion http-version]
  (case (.getMajor http-version)
    3 :http3
    2 :http2
    1 (keyword (str "http-1." (.getMinor http-version)))))

(defn- response-map [^Message response
                     {:keys [request-map
                             ^ClassicHttpRequest request
                             http-client]}]
  (let [response-head ^HttpResponse (.getHead response)]
    {:uri         (.getUri request)
     :status      (.getCode response-head)
     :body        (.getBody response)
     :headers     (->> (.getHeaders response-head)
                       (into {} (map (juxt (comp str/lower-case Header/.getName) Header/.getValue))))
     :version     (-> (.getVersion response-head)
                      http-version->kw)
     :http-client http-client
     :request     (assoc request-map :http-request request)}))

(defn- entity-consumer ^AsyncEntityConsumer [stream-buffer-size executor-service]
  (proxy [AbstractClassicEntityConsumer] [stream-buffer-size executor-service]
    (consumeData [_content-type input-stream]
      input-stream)))

(declare write-input-stream!)

(deftype InputStreamEntityProducer [^InputStream input-stream ex-atom stream-buffer-size content-length chunking-strategy]
  AsyncEntityProducer
  (isRepeatable [_this] false)
  (isChunked [_this] (or (nil? content-length)
                         (neg? content-length)
                         (> content-length stream-buffer-size)))
  (getContentLength [_this] (or content-length -1))
  (getContentType [_this] nil)
  (getContentEncoding [_this] nil)
  (getTrailerNames [_this] #{})
  (releaseResources [_this] (.close input-stream))
  (available [_this] (.available input-stream))
  (failed [this ex]
    (when (compare-and-set! ex-atom nil ex)
      (.releaseResources this)))
  (^void produce [this ^DataStreamChannel dsc]
    (with-open [_ this]
      (if (.isChunked this)
        (write-input-stream! chunking-strategy this dsc)
        (let [ba (if (pos? (.getContentLength this))
                   (.readNBytes input-stream content-length)
                   (.readAllBytes input-stream))]
          (->> (ByteBuffer/wrap ba)
               (.write dsc))))
      (.endStream dsc)))
  Closeable
  (^void close [this] (.releaseResources this)))

(defmulti write-input-stream! (fn [chunking-strategy ^InputStreamEntityProducer _ ^DataStreamChannel _] chunking-strategy))

(defmethod write-input-stream! :default [_ p dsc]
  (write-input-stream! ::upstream-chunking p dsc))

;write chunks based on input-stream chunks
(defmethod write-input-stream! ::upstream-chunking [_ ^InputStreamEntityProducer p ^DataStreamChannel dsc]
  (let [stream-buffer-size (.-stream-buffer-size p)
        ^InputStream input-stream (.-input-stream p)
        ba (byte-array stream-buffer-size)]
    (loop []
      (let [size (.read ^InputStream input-stream ^bytes ba)]
        (when (pos? size)
          (->> (ByteBuffer/wrap ba 0 size)
               (.write dsc))
          (recur))))))

;try to minimise number of chunks
(defmethod write-input-stream! ::minimal-chunking [_ ^InputStreamEntityProducer p ^DataStreamChannel dsc]
  (let [stream-buffer-size (.-stream-buffer-size p)
        ^InputStream input-stream (.-input-stream p)
        ba (byte-array stream-buffer-size)]
    (loop [offset 0]
      (let [size (.read input-stream ^bytes ba offset (- stream-buffer-size offset))
            read (+ offset size)]
        (cond
          (= read stream-buffer-size) (do (->> (ByteBuffer/wrap ba 0 read) ;buffer is full
                                               (.write dsc))
                                          (recur 0))
          (pos? size) (recur read) ;read from input-stream but buffer is not yet full
          (pos? read) (->> (ByteBuffer/wrap ba 0 read) ;input-stream is finished and buffer has unwritten content
                           (.write dsc)))))))

(defn- input-stream-entity-producer ^AsyncEntityProducer [^InputStream is stream-buffer-size content-length chunking-strategy]
  (->InputStreamEntityProducer is (atom nil) stream-buffer-size content-length chunking-strategy))

(defn- entity-producer ^AsyncEntityProducer [body stream-buffer-size content-length chunking-strategy]
  (cond
    (nil? body) nil
    (instance? InputStream body) (input-stream-entity-producer body stream-buffer-size content-length chunking-strategy)
    (string? body) (StringAsyncEntityProducer. ^String body nil)
    (bytes? body) (BasicAsyncEntityProducer. ^bytes body nil)
    :else (throw (IllegalArgumentException. (str "Body not supported: " (class body))))))

(defn- completable-future-callback [^CompletableFuture cf request-details]
  (reify FutureCallback
    (completed [_ v]
      (->> (response-map v request-details)
           (.complete cf)))
    (failed [_ ex]
      (.completeExceptionally cf ex))
    (cancelled [_]
      (.cancel cf true))))

(defrecord ApacheHttpClient [^CloseableHttpAsyncClient client-impl
                             ^ExecutorService executor-service
                             stream-buffer-size]
  hcp/HttpClient
  (hcp/client-type [_] :apache)
  (hcp/request* [this {:keys [request-method version headers body timeout async? content-length] :as req} respond raise]
    (let [version (->http-version version)
          method (or (some-> request-method name str/upper-case)
                     "GET")
          request (-> (ClassicRequestBuilder/create method)
                      (.setUri (req-uri req))
                      (.setVersion version)
                      (add-headers! headers)
                      .build)
          request-details {:request-map req
                           :request     request
                           :http-client this}
          producer (BasicRequestProducer. request (entity-producer body stream-buffer-size content-length
                                                                   (::chunking-strategy req)))
          consumer (BasicResponseConsumer. (entity-consumer stream-buffer-size executor-service))]
      (if-not async?
        (-> (.execute client-impl producer consumer nil nil nil)
            (.get timeout TimeUnit/MILLISECONDS)
            (response-map request-details))
        (let [callback-cf (CompletableFuture.)
              cf (-> (.thenApply callback-cf ^Function respond)
                     (.exceptionally ^Function raise)
                     (.orTimeout ^long timeout TimeUnit/MILLISECONDS))]
          (->> (completable-future-callback callback-cf request-details)
               (.execute client-impl producer consumer nil nil))
          cf))))
  Closeable
  (^void close [_]
    (.close client-impl)
    (.close executor-service)))

(defdelayed default-client-impl (doto (HttpAsyncClients/createDefault)
                                  (.start)))

(defdelayed default-http-client
  (->ApacheHttpClient
    (default-client-impl)
    (Executors/newVirtualThreadPerTaskExecutor)
    4096))

(comment
  (->http-version :http-1.1)
  (hcp/client-type (default-http-client))
  (-> (hcp/request*
        (default-http-client)
        {:scheme         "http"
         :server-name    "localhost"
         :server-port    8084
         :timeout        10000
         :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
         :request-method :post
         ;:version        :http2
         :async?         true
         :headers        {"test" "header"}
         :body           "my body"}
        (fn [x] (println x) x)
        #(throw %))
      .get
      (update :body slurp))
  (hcp/request*
    (default-http-client)
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    8084
     :timeout        10000
     :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
     :request-method :post
     :headers        {"test" "header"}
     :body           "my body"}
    nil nil)
  (hcp/request*
    (default-http-client)
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    2226
     :timeout        10000
     :uri            "/tenants/961478ea-4c14-4830-8f50-b504965adfe0/alerts"
     :request-method :get
     :version        :http-1.1}
    nil nil)
  (hcp/request*
    (default-http-client)
    {:scheme         "https"
     :server-name    "gitlab.com"
     ;:server-port    2999
     ;:uri            "/test/abc/error2"
     :request-method :get
     :timeout        10000
     ;:headers        {"test" "header"}
     :body           "my body"}
    nil nil))
