(ns prism.http
  (:require
    [clojure.java.io :as io]
    [hato.client :as hato]
    [hato.conversion :as hc]
    [hato.middleware :as hmw]
    [hato.multipart :as multipart]
    [prism.core :refer [defdelayed] :as prism]
    [prism.http-client-protocol :as hcp]
    [prism.internal.aws-sign :as aws-sign]
    [prism.internal.classpath :as cp])
  (:import
    (java.io InputStream Reader)
    (java.net ConnectException)
    (java.net.http HttpRequest$BodyPublishers)
    (java.util.concurrent Flow$Publisher)))

(defn- stream-request->publisher-request [{:keys [body content-length] :as req}]
  (if (instance? InputStream body)
    (let [req (assoc req :body (HttpRequest$BodyPublishers/ofInputStream (fn [] body)))]
      (cond-> req
              (and content-length
                   (pos? content-length)) (update :body ^[Flow$Publisher long] HttpRequest$BodyPublishers/fromPublisher content-length)))
    req))

(defrecord DefaultHatoClient []
  hcp/HttpClient
  (hcp/client-type [_] :hato)
  (hcp/request* [_ req-map respond raise]
    (let [req-map (stream-request->publisher-request req-map)]
      (if (:async? req-map)
        (hato/request* req-map #(respond (update % :headers dissoc ":status")) raise)
        (-> (hato/request* req-map)
            (update :headers dissoc ":status"))))))

(def ^:private implementation
  (or (cp/when-ns 'prism.apache-http-client
        prism.apache-http-client/default-http-client)
      ->DefaultHatoClient))

(defdelayed default-http-client (implementation))

(def ^:private wrap-retry-request nil)
(declare ^:private attempt-request)
(cp/when-ns 'again.core
  (cp/when-ns 'taoensso.timbre
    (let [cb (fn [{:again.core/keys [exception status]}]
               (when (#{:retry :attempt} status)
                 (let [data (ex-data exception)
                       request (-> data :request (select-keys [:url :method]))
                       response (-> data (select-keys [:status :body]))]
                   (taoensso.timbre/with-context
                     (merge request response)
                     (taoensso.timbre/warnf exception "Error while making HTTP request."))))
               (when (or (-> exception
                             ex-data
                             :status
                             (= 400))
                         (instance? ConnectException exception))
                 :again.core/fail))
          retry-config (fn [{request-retry :retry}]
                         {:again.core/callback cb
                          :again.core/strategy (cond
                                                 (vector? request-retry) request-retry
                                                 (number? request-retry) (take request-retry (list* 1000 2000 (repeat 3000)))
                                                 (true? request-retry) [1000 2000 3000 3000])})]

      (defn- wrap-retry-request [client]
        (fn
          ([req]
           (if (:retry req)
             (again.core/with-retries
               (retry-config req)
               (client req))
             (client req)))
          ([req respond raise]
           (if (:retry req)
             (client req
                     #(again.core/with-retries
                        (retry-config req)
                        (respond %))
                     raise)
             (client req respond raise))))))))

(defn- attempt-request [f req]
  (try
    (f req)
    (catch Exception e
      (cp/when-ns 'taoensso.timbre
        (taoensso.timbre/debugf "HTTP Exception: %s" e))
      e)))

(defn- wrap-catch-exceptions [client]
  (fn
    ([req]
     (if (:catch-exceptions? req)
       (attempt-request client req)
       (client req)))
    ([req respond raise]
     (if (:catch-exceptions? req)
       (client req #(attempt-request respond %) raise)
       (client req respond raise)))))

(defn- default-user-agent [req]
  (update-in req [:headers "user-agent"]
             (fnil identity (str "Prism " (prism/prism-version)))))

(defn- wrap-user-agent [client]
  (fn
    ([req]
     (-> (default-user-agent req)
         (client)))
    ([req respond raise]
     (client (default-user-agent req) respond raise))))

(defn- resp->charset [resp] (or (-> resp :content-type-params :charset) "UTF-8"))

(defn- coerce-response-body [{:keys [coerce]} {:keys [body status] :as resp} coerce-fn]
  (let [^String charset (resp->charset resp)]
    (cond
      (and (hmw/unexceptional-status? status)
           (or (nil? coerce) (= coerce :unexceptional)))
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      (= coerce :always)
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      (and (not (hmw/unexceptional-status? status)) (= coerce :exceptional))
      (with-open [r (io/reader body :encoding charset)]
        (assoc resp :body (coerce-fn r)))
      :else resp)))

(def ^:private wrap-json-body nil)
(cp/when-ns 'prism.json
  (defn- coerce-json-response-body [req resp]
    (coerce-response-body req resp prism.json/json->clj))
  (defmethod hmw/coerce-response-body :json [req resp] (coerce-json-response-body req resp))
  (defmethod hmw/coerce-response-body :json-strict [req resp] (coerce-json-response-body req resp))

  (defmethod hc/decode :application/json
    [resp _]
    (let [^String charset (resp->charset resp)]
      (-> (:body resp)
          (io/reader :encoding charset)
          prism.json/json->clj)))

  (defn- wrap-json-body [client]
    (let [add-json-body (fn [req]
                          (if-some [body-json (:json req)]
                            (-> (assoc req :body (prism.json/write-json-string body-json)
                                           :content-type :json)
                                (dissoc :json))
                            req))]
      (fn
        ([req]
         (-> (add-json-body req)
             client))
        ([req respond raise]
         (-> (add-json-body req)
             (client respond raise)))))))

(cp/when-class 'org.json.XML
  (declare ^:private xml->map)
  (defn- xml->val [xml-val]
    (cond
      (= org.json.JSONObject/NULL xml-val) nil
      (instance? org.json.JSONObject xml-val) (xml->map xml-val)
      (instance? org.json.JSONArray xml-val) (mapv xml->val xml-val)
      :else xml-val))

  (defn- xml->map [^org.json.JSONObject jo]
    (-> (reduce
          (fn [m k]
            (->> (.get jo k)
                 xml->val
                 (assoc!
                   m
                   (keyword k))))
          (transient {})
          (.keySet jo))
        persistent!))

  (defn- xml->clj [r]
    (let [jo (org.json.XML/toJSONObject ^Reader r)]
      (xml->map jo)))

  (defmethod hmw/coerce-response-body :xml [req resp]
    (coerce-response-body req resp xml->clj))

  (defmethod hc/decode :application/xml
    [resp _]
    (let [^String charset (resp->charset resp)]
      (-> (:body resp)
          (io/reader :encoding charset)
          xml->map))))

(defn- sign-aws-request [request]
  (if-let [aws-config (:aws request)]
    (-> (aws-sign/sign-request request aws-config)
        (dissoc :aws))
    (dissoc request :aws)))

(defn- wrap-aws-sign [client]
  (fn
    ([request]
     (-> (sign-aws-request request)
         client))
    ([request respond raise]
     (-> (sign-aws-request request)
         (client respond raise)))))

(defn- remove-hop-to-hop-headers [response]
  (update
    response
    :headers
    dissoc
    ;https://datatracker.ietf.org/doc/html/rfc2616#section-13.5.1
    "connection"
    "keep-alive"
    "proxy-authenticate"
    "proxy-authorization"
    "te"
    "trailers"
    "transfer-encoding"
    "upgrade"))

(defn- proxy-request [request]
  (-> (update request :as (fnil identity :stream))
      (update :accept (fnil identity "*/*"))
      (update :decompress-body (fnil identity false))
      (update :throw-exceptions (fnil identity false))
      (update :remove-hop-to-hop-headers? (fnil identity true))))

(defn- wrap-proxy [client]
  (fn
    ([request]
     (-> (dissoc request :proxy?)
         (cond-> (:proxy? request) proxy-request)
         client))
    ([request respond raise]
     (-> (dissoc request :proxy?)
         (cond-> (:proxy? request) proxy-request)
         (client respond raise)))))

(defn- wrap-remove-hop-to-hop-headers [client]
  (fn
    ([request]
     (-> (client request)
         (cond-> (:remove-hop-to-hop-headers? request) remove-hop-to-hop-headers)))
    ([request respond raise]
     (client request
             (cond-> respond
                     (:remove-hop-to-hop-headers? request) (comp remove-hop-to-hop-headers))
             raise))))

(defn- defaulted-http-client-request [request]
  (update request :http-client (fnil identity (default-http-client))))

(defn- wrap-add-http-client [client]
  (fn
    ([request]
     (client (defaulted-http-client-request request)))
    ([request respond raise]
     (-> (defaulted-http-client-request request)
         (client respond raise)))))

(defn- multipart-request
  "Based on hato.middleware/multipart-request"
  [{:keys [multipart] :as req}]
  (if multipart
    (let [b (multipart/boundary)
          segments (multipart/raw-multipart-payload-segments multipart b)
          input-stream (multipart/body segments)
          content-length (multipart/content-length segments)]
      (-> (dissoc req :multipart)
          (assoc :body input-stream)
          (assoc :content-length content-length)
          (update :headers assoc "content-type" (str "multipart/form-data; boundary=" b))))
    req))

(defn- wrap-multipart [client]
  (fn
    ([request]
     (client (multipart-request request)))
    ([request respond raise]
     (client (multipart-request request) respond raise))))

(def ^:private wrapped-request
  (->> [hmw/wrap-request-timing ;called last before request is made, called first after request is made

        wrap-aws-sign

        hmw/wrap-query-params
        hmw/wrap-basic-auth
        hmw/wrap-oauth
        hmw/wrap-user-info
        hmw/wrap-url

        hmw/wrap-decompression
        hmw/wrap-output-coercion
        hmw/wrap-exceptions
        hmw/wrap-accept
        hmw/wrap-accept-encoding
        wrap-multipart
        ;hmw/wrap-multipart ;not included due to its use of HttpRequest$BodyPublishers

        hmw/wrap-content-type
        hmw/wrap-form-params
        hmw/wrap-nested-params
        hmw/wrap-method

        wrap-add-http-client
        wrap-json-body
        wrap-retry-request
        wrap-user-agent
        wrap-remove-hop-to-hop-headers
        wrap-proxy
        wrap-catch-exceptions]
       (filterv some?)
       (hmw/wrap-request hcp/variable-client-request*)))

(defn request [req]
  (let [req (update req :timeout (fnil identity 10000))]
    (if (:async? req)
      (wrapped-request req (:respond req identity) (:raise req #(throw %)))
      (wrapped-request req))))

(comment
  (defn- gen-string [size]
    (apply str (repeatedly size #(rand-nth "abcdefghijklmnopqrstuvwxyz0123456789"))))
  (do (request {:url     "https://webhook.site/1db28338-bb65-450c-9132-04f0feb29cb8"
                :method  :post
                :timeout Long/MAX_VALUE
                :body    (java.io.StringBufferInputStream. (gen-string 19999 #_20000))
                :headers {}})
      nil)
  (do (request {:url          "http://localhost:8084/6a2cdd5a-c081-4679-a68a-a2075301f937"
                :query-params {:client "hato"}
                :method       :post
                :http-client  (->DefaultHatoClient)
                :body         (java.io.StringBufferInputStream. (gen-string 1e4))
                :headers      {}})
      (request {:url          "http://localhost:8084/6a2cdd5a-c081-4679-a68a-a2075301f937"
                :query-params {:client "default"}
                :method       :post
                :body         (java.io.StringBufferInputStream. (gen-string 1e4))
                :headers      {}})
      nil)
  (request {:url          "https://github.com"
            :method       :get
            :proxy?       true
            :query-params {:query "xyz"}}))
