(ns prism.internal.pg-extensions
  (:require
    [clojure.string :as s]
    [honey.sql :as hsql]
    [honey.sql.protocols :as hsp]
    [next.jdbc.date-time :as date-time]
    [next.jdbc.prepare :refer [SettableParameter]]
    [next.jdbc.result-set :refer [ReadableColumn]]
    [prism.internal.classpath :as cp])
  (:import
    (clojure.lang IObj IPersistentList IPersistentMap IPersistentSet IPersistentVector ISeq Sequential)
    (java.sql PreparedStatement)
    (java.sql PreparedStatement Timestamp)
    (java.time Instant)
    (java.util Date UUID)
    (java.util.concurrent TimeUnit)
    (org.postgresql.util PGobject)))

(defn- format-greptime-with [_ withs]
  (let [{:keys [clause params]} (reduce-kv
                                  (fn [acc k v]
                                    (let [[sql param] (hsql/format-expr v)
                                          k (hsql/format-entity k)]
                                      (-> (update acc :clause str k "=" sql ",")
                                          (cond-> (some? param) (update :params conj param)))))
                                  {:clause "with("
                                   :params []}
                                  withs)
        clause (-> (subs clause 0 (dec (count clause)))
                   (str ")"))]
    (into [clause] params)))

(defn- format-greptime-engine [_ engine]
  [(str "engine=" engine)])

(defn format-greptime-interval [_ interval]
  [(str "INTERVAL(" (hsql/format-entity interval) ")")])

(defn format-greptime-set-ttl [_ ttl]
  [(str "SET 'ttl'='" ttl \')])

(defn- sqlize-coll [coll]
  (as-> (mapv hsp/sqlize coll) $
        (s/join ", " $)
        (str \( $ \))))

(defn- json-get-str [_ [& exprs]]
  (let [[sqls params] (hsql/format-expr-list exprs)
        sql (-> (repeat (- (count sqls) 2) " -> ")
                vec
                (into [" ->> " nil])
                (->> (interleave sqls))
                s/join)]
    (into [sql] params)))

(defn extend-protocols! []
  ;general postgres type handling changes
  (date-time/read-as-instant)
  (hsql/register-fn! :->> json-get-str)
  (hsql/register-op! :-> :variadic true)
  (hsql/register-op! :||)

  (extend-protocol SettableParameter
    Date
    (set-parameter [v ^PreparedStatement s i]
      (.setTimestamp s i (-> v
                             (inst-ms)
                             (Timestamp.))))

    Instant
    (set-parameter [v ^PreparedStatement s i]
      (.setTimestamp s i (Timestamp/from v)))

    String
    (set-parameter [v ^PreparedStatement s ^long i]
      (let [param-type (-> (.getParameterMetaData s)
                           (.getParameterType i))]
        (.setObject s i v ^int param-type))))

  (extend-protocol ReadableColumn
    UUID
    (read-column-by-label [v _]
      (str v))
    (read-column-by-index [v _ _]
      (str v)))

  ;postgres json
  (cp/when-ns 'prism.json
    (letfn [(->pgobject [x]
              (let [pg-type (or (:pg-type (meta x)) "jsonb")]
                (doto (PGobject.)
                  (.setType pg-type)
                  (.setValue (prism.json/write-json-string x)))))
            (<-pgobject
              [^PGobject v]
              (let [type (.getType v)
                    value (.getValue v)]
                (if (#{"jsonb" "json"} type)
                  (let [v (prism.json/json->clj value)]
                    (cond-> v
                            (instance? IObj v) (with-meta {:pg-type type})))
                  value)))]
      (extend-protocol SettableParameter
        IPersistentMap
        (set-parameter [m ^PreparedStatement s i]
          (.setObject s i (->pgobject m)))

        IPersistentVector
        (set-parameter [v ^PreparedStatement s i]
          (.setObject s i (->pgobject v))))

      (extend-protocol ReadableColumn
        PGobject
        (read-column-by-label [^PGobject v _]
          (<-pgobject v))
        (read-column-by-index [^PGobject v _ _]
          (<-pgobject v)))))

  ;greptime handling
  (extend-protocol hsp/InlineValue
    Sequential
    (sqlize [this] (sqlize-coll this))
    IPersistentVector
    (sqlize [this] (sqlize-coll this))
    IPersistentList
    (sqlize [this] (sqlize-coll this))
    IPersistentSet
    (sqlize [this] (sqlize-coll this))
    ISeq
    (sqlize [this] (sqlize-coll this)))

  (hsql/register-clause! :insert-into :insert-into :select)
  (hsql/register-clause! :greptime/with-metadata format-greptime-with nil)
  (hsql/register-clause! :greptime/engine format-greptime-engine :greptime/with-metadata)
  (hsql/register-clause! :greptime/tags :values :join-by)
  (hsql/register-clause! :greptime/interval format-greptime-interval :order-by)
  (hsql/register-clause! :greptime/set-ttl format-greptime-set-ttl :add-column)
  (hsql/register-clause! :greptime/as-select :select :create-table)
  (hsql/register-clause! :greptime/sink-to :insert-into :greptime/as-select)
  (hsql/register-clause! :greptime/create-flow :insert-into :greptime/sink-to)
  (hsql/register-fn! :greptime/range
                     (fn format-greptime-range [_fn [expr interval]]
                       (let [[sql & params] (hsql/format-expr expr)]
                         (into [(str sql " RANGE '" interval "'")] params))))

  (hsql/register-clause! :greptime/align
                         (fn format-greptime-align [_ [interval timestamp-expr columns]]
                           (let [[timestamp-sql & params] (when timestamp-expr
                                                            (hsql/format-expr timestamp-expr))]
                             (into
                               [(str "ALIGN '" interval "'"
                                     (when timestamp-sql (str " TO " timestamp-sql ""))
                                     (when columns
                                       (str " BY " (str "(" (s/join ", " columns) ")"))))]
                               params)))
                         :order-by))

