(ns prism.apache-http-client
  (:require
    [clojure.string :as str]
    [prism.core :refer [defdelayed]])
  (:import
    (java.io Closeable InputStream)
    (java.net URI)
    (java.nio ByteBuffer)
    (java.util.concurrent CompletableFuture Executor Executors TimeUnit)
    (java.util.function Function)
    (org.apache.hc.client5.http.async HttpAsyncClient)
    (org.apache.hc.client5.http.impl.async HttpAsyncClients)
    (org.apache.hc.core5.concurrent FutureCallback)
    (org.apache.hc.core5.http ClassicHttpRequest Header HttpResponse HttpVersion Message)
    (org.apache.hc.core5.http.io.support ClassicRequestBuilder)
    (org.apache.hc.core5.http.nio AsyncEntityConsumer AsyncEntityProducer DataStreamChannel)
    (org.apache.hc.core5.http.nio.entity BasicAsyncEntityProducer StringAsyncEntityProducer)
    (org.apache.hc.core5.http.nio.support BasicRequestProducer BasicResponseConsumer)
    (org.apache.hc.core5.http.nio.support.classic AbstractClassicEntityConsumer)))

(set! *warn-on-reflection* true)

(defdelayed ^HttpAsyncClient default-client (doto (HttpAsyncClients/createDefault)
                                              (.start)))
(def ^:private stream-buffer-size 4096)
(defdelayed ^:private ^Executor virtual-threads-executor (Executors/newVirtualThreadPerTaskExecutor))

(defn- add-headers! ^ClassicRequestBuilder [r headers]
  (reduce-kv ^[String String] ClassicRequestBuilder/.addHeader r headers))

(defn- req-uri "As per hato.client/ring-request->HttpRequest"
  ^URI [{:keys [scheme server-name server-port uri query-string]}]
  (-> (str (name scheme)
           "://"
           server-name
           (some->> server-port (str ":"))
           uri
           (some->> query-string (str "?")))
      URI.))

(defn- ->http-version ^HttpVersion [kw]
  (case kw
    :http-1.1 HttpVersion/HTTP_1_1
    :http2 HttpVersion/HTTP_2_0
    nil))

(defn- http-version->kw [^HttpVersion http-version]
  (case (.getMajor http-version)
    3 :http3
    2 :http2
    1 (keyword (str "http-1." (.getMinor http-version)))))

(defn- response-map [^Message response
                     {:keys [request-map
                             ^ClassicHttpRequest request
                             http-client]}]
  (let [response-head ^HttpResponse (.getHead response)]
    {:uri         (.getUri request)
     :status      (.getCode response-head)
     :body        (.getBody response)
     :headers     (->> (.getHeaders response-head)
                       (into {} (map (juxt (comp str/lower-case Header/.getName) Header/.getValue))))
     :version     (-> (.getVersion response-head)
                      http-version->kw)
     :http-client http-client
     :request     (assoc request-map :http-request request)}))

(defn- entity-consumer ^AsyncEntityConsumer []
  (proxy [AbstractClassicEntityConsumer] [stream-buffer-size (virtual-threads-executor)]
    (consumeData [_content-type input-stream]
      input-stream)))

(deftype InputStreamEntityProducer [^InputStream input-stream ex-atom]
  AsyncEntityProducer
  (isRepeatable [_this] false)
  (isChunked [_this] false)
  (getContentLength [_this] -1)
  (getContentType [_this] nil)
  (getContentEncoding [_this] nil)
  (getTrailerNames [_this] #{})
  (releaseResources [_this] (.close input-stream))
  (available [_this] (.available input-stream))
  (failed [this ex]
    (when (compare-and-set! ex-atom nil ex)
      (.releaseResources this)))
  (^void produce [this ^DataStreamChannel dsc]
    (with-open [_ this]
      (let [ba (byte-array stream-buffer-size)]
        (loop []
          (let [size (.read input-stream ^bytes ba)]
            (when (pos? size)
              (->> (ByteBuffer/wrap ba 0 size)
                   (.write dsc))
              (recur)))))
      (.endStream dsc)))
  Closeable
  (^void close [this] (.releaseResources this)))

(defn- input-stream-entity-producer ^AsyncEntityProducer [^InputStream is]
  (->InputStreamEntityProducer is (atom nil)))

(defn- entity-producer ^AsyncEntityProducer [body]
  (cond
    (nil? body) nil
    (instance? InputStream body) (input-stream-entity-producer body) ;TODO handle content-length?
    (string? body) (StringAsyncEntityProducer. ^String body nil)
    (bytes? body) (BasicAsyncEntityProducer. ^bytes body nil)))

(defn- completable-future-callback [^CompletableFuture cf request-details]
  (reify FutureCallback
    (completed [_ v]
      (->> (response-map v request-details)
           (.complete cf)))
    (failed [_ ex]
      (.completeExceptionally cf ex))
    (cancelled [_]
      (.cancel cf true))))

(defn request*
  ([req-map] (request* req-map nil nil))
  ([{:keys [request-method version headers body http-client timeout async?] :as req} respond raise]
   (let [version (->http-version version)
         method (or (some-> request-method name str/upper-case)
                    "GET")
         request (-> (ClassicRequestBuilder/create method)
                     (.setUri (req-uri req))
                     (.setVersion version)
                     (add-headers! headers)
                     .build)
         ^HttpAsyncClient client (or http-client (default-client))
         request-details {:request-map req
                          :request     request
                          :http-client client}
         producer (BasicRequestProducer. request (entity-producer body))
         consumer (BasicResponseConsumer. (entity-consumer))]
     (if-not async?
       (-> (.execute client producer consumer nil nil nil)
           (.get timeout TimeUnit/MILLISECONDS)
           (response-map request-details))
       (let [callback-cf (CompletableFuture.)
             cf (-> (.thenApply callback-cf ^Function respond)
                    (.exceptionally ^Function raise)
                    (.orTimeout ^long timeout TimeUnit/MILLISECONDS))]
         (->> (completable-future-callback callback-cf request-details)
              (.execute client producer consumer nil nil))
         cf)))))

(comment
  (->http-version :http-1.1)
  (-> (request*
        {:scheme         "http"
         :server-name    "localhost"
         :server-port    8084
         :timeout        10000
         :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
         :request-method :post
         ;:version        :http2
         :async?         true
         :headers        {"test" "header"}
         :body           "my body"}
        (fn [x] (println x) x)
        #(throw %))
      .get
      (update :body slurp))
  (request*
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    8084
     :timeout        10000
     :uri            "/6a2cdd5a-c081-4679-a68a-a2075301f937"
     :request-method :post
     :headers        {"test" "header"}
     :body           "my body"})
  (request*
    {:scheme         "http"
     :server-name    "localhost"
     :server-port    2226
     :timeout        10000
     :uri            "/tenants/961478ea-4c14-4830-8f50-b504965adfe0/alerts"
     :request-method :get
     :version        :http-1.1})
  (request*
    {:scheme         "https"
     :server-name    "gitlab.com"
     ;:server-port    2999
     ;:uri            "/test/abc/error2"
     :request-method :get
     :timeout        10000
     ;:headers        {"test" "header"}
     :body           "my body"}))
