(ns prism.internal.http-client-helidon
  (:require
    [clojure.string :as s]
    [prism.core :refer [defdelayed]])
  (:import
    (clojure.lang ExceptionInfo IPersistentCollection)
    (io.helidon.common GenericType)
    (io.helidon.faulttolerance Async)
    (io.helidon.http Header HeaderNames Headers Method WritableHeaders)
    (io.helidon.http.media EntityWriter MediaSupport MediaSupport$SupportLevel MediaSupport$WriterResponse)
    (io.helidon.webclient.api ClientRequest HttpClientRequest WebClient WebClientConfig$Builder)
    (io.helidon.webclient.http1 Http1ClientResponse)
    (io.helidon.webclient.http2 Http2ClientResponse)
    (java.io InputStream OutputStream)
    (java.net URI)
    (java.time Duration)
    (java.util Collection OptionalLong)
    (java.util.function Function)
    (prism.internal HttpResponseDelegate)))

(defprotocol ^:private ->Array
  (->array ^"[Ljava.lang.String;" [_]))

(extend-protocol ->Array
  String
  (->array [this] (into-array String [this]))
  Collection
  (->array [^Collection this] (.toArray this))
  IPersistentCollection
  (->array [this] (into-array String this)))

(defprotocol ^:private HttpVersion
  (http-version [_]))

(extend-protocol HttpVersion
  Http1ClientResponse
  (http-version [_] :http-1.1)
  Http2ClientResponse
  (http-version [_] :http2))

(def http-methods
  {:get     Method/GET
   :post    Method/POST
   :put     Method/PUT
   :delete  Method/DELETE
   :head    Method/HEAD
   :options Method/OPTIONS
   :trace   Method/TRACE
   :patch   Method/PATCH
   :connect Method/CONNECT})

(def ^:private max-buffer-size 1024)

(defn- feed-stream
  ([input output content-length]
   (with-open [^InputStream input input
               ^OutputStream output output]
     (let [buf (byte-array (min content-length max-buffer-size))]
       (loop [size (.read input buf)
              remaining (- content-length size)]
         (when (pos? size)
           (.write output buf)
           (when (pos? remaining)
             (let [size (.read input buf)]
               (recur
                 size
                 (- content-length size)))))))))
  ([input output]
   (with-open [^InputStream input input
               ^OutputStream output output]
     (let [buf (byte-array max-buffer-size)]
       (loop [size (.read input buf)]
         (when (pos? size)
           (.write output buf)
           (recur (.read input buf))))))))

(defn- write-input-stream! [input output ^OptionalLong content-length]
  (if-let [content-length (when (.isPresent content-length)
                            (.getAsLong content-length))]
    (feed-stream input output content-length)
    (feed-stream input output)))

(defdelayed ^:private input-stream-writer
  (reify EntityWriter
    (^void write [_
                  ^GenericType _
                  input
                  ^OutputStream output
                  ^Headers headers ^WritableHeaders _mut-headers]
      (write-input-stream!
        input output
        (.contentLength headers)))
    (^void write [_
                  ^GenericType _
                  input
                  ^OutputStream output
                  ^WritableHeaders headers]
      (write-input-stream!
        input output
        (.contentLength headers)))))

(defn- supported-input-stream-writer [^GenericType type]
  (if (.isAssignableFrom InputStream (.rawType type))
    (MediaSupport$WriterResponse. MediaSupport$SupportLevel/COMPATIBLE input-stream-writer)
    (MediaSupport$WriterResponse/unsupported)))

(defdelayed ^:private input-stream-media-support
  (reify MediaSupport
    (name [_] "input-stream")
    (type [_] "input-stream")
    (^MediaSupport$WriterResponse writer [_ ^GenericType type ^WritableHeaders _]
      (supported-input-stream-writer type))
    (^MediaSupport$WriterResponse writer [_ ^GenericType type ^Headers _ ^WritableHeaders _]
      (supported-input-stream-writer type))))

(defdelayed ^WebClient default-client (-> (WebClient/builder)
                                          ^WebClientConfig$Builder (.addMediaSupport (input-stream-media-support))
                                          .build))

(defn- add-headers! ^HttpClientRequest [^HttpClientRequest r m]
  (reduce-kv
    (fn add-header! [^ClientRequest r k v]
      (.header
        r
        (HeaderNames/create (name k))
        (->array (or v ""))))
    r
    m))

(defn- headers->map [headers]
  (persistent!
    (reduce
      (fn assoc-header! [acc ^Header header]
        (let [values (.allValues header)]
          (assoc! acc
                  (-> (.name header) s/lower-case)
                  (if (= (.size values) 1)
                    (.get values 0)
                    (vec values)))))
      (transient {})
      headers)))

(defn- execute-request! ^HttpResponseDelegate [^HttpClientRequest helidon-request {:keys [body] :as req}]
  (try
    (-> (if body
          (.submit helidon-request body)
          (.request helidon-request))
        HttpResponseDelegate.)
    (catch Exception e
      (throw (ex-info "Failed to execute request" req e)))))

(defn- req-uri "As per hato.client/ring-request->HttpRequest"
  ^URI [{:keys [scheme server-name server-port uri query-string]}]
  (-> (str (name scheme)
           "://"
           server-name
           (some->> server-port (str ":"))
           uri
           (some->> query-string (str "?")))
      URI.))

(defn- helidon-method ^Method [request-method]
  (or (get http-methods request-method)
      (some-> request-method name Method/create)
      Method/GET))

(defn- response-map [{:keys [request-map
                             ^HttpClientRequest request
                             ^HttpResponseDelegate response
                             http-client]}]
  {:uri         (.uri request)
   :status      (.. response status code)
   :body        (.inputStream response)
   :headers     (-> (.headers response)
                    headers->map)
   :version     (http-version (.delegate response))
   :http-client http-client
   :request     (assoc request-map :http-request request)})

(defn- raise-fn [raise]
  (fn [e]
    (let [cause (.getCause ^Exception e)]
      (if (instance? ExceptionInfo cause)
        (raise cause)
        (raise e)))))

(defn- invoke-async [{:keys [request-map
                             ^HttpClientRequest request]
                      :as   request-details}
                     respond raise]
  (-> (Async/invokeStatic #(execute-request! request request-map))
      (.thenApply #(-> (assoc request-details :response %)
                       response-map
                       respond))
      (.exceptionally ^Function (raise-fn raise))))

(defn request [{:keys [http-client request-method headers version timeout async?] :as req}
               & [respond raise]]
  (let [^WebClient client (or http-client (default-client))
        helidon-request (-> (.method client (helidon-method request-method))
                            (.uri (req-uri req))
                            (add-headers! headers)
                            (.protocolId (case version
                                           :http2 "h2"
                                           :http-1.1 "http/1.1"
                                           nil))
                            (.readTimeout (Duration/ofMillis timeout)))]
    (if-not async?
      (response-map
        {:request-map req
         :request     helidon-request
         :response    (execute-request! helidon-request req)
         :http-client client})
      (invoke-async
        {:request-map req
         :request     helidon-request
         :http-client client}
        respond
        raise))))

(comment
  (dotimes [i 1]
    (request
      {:headers        {"time"            (str (java.time.Instant/now))
                        "index"           (str i)
                        "user-agent"      "Prism local"
                        "accept-encoding" "gzip, deflate"}
       :server-port    nil
       :url            "https://webhook.site/f83b55ad-9f22-4ede-900b-7863556b1faf"
       :uri            "/f83b55ad-9f22-4ede-900b-7863556b1faf"
       :server-name    "webhook.site"
       :timeout        10000
       :query-string   "abc=a+test"
       :scheme         :https
       :version        :http2
       :async?         true
       :request-method :get})))
