(ns prism.internal.http-client-helidon
  (:require
    [clojure.string :as s]
    [prism.core :refer [defdelayed]])
  (:import
    (clojure.lang IPersistentCollection)
    (io.helidon.http Header HeaderNames Method)
    (io.helidon.webclient.api ClientRequest HttpClientRequest WebClient)
    (io.helidon.webclient.http1 Http1ClientResponse)
    (io.helidon.webclient.http2 Http2ClientResponse)
    (java.net URI)
    (java.time Duration)
    (java.util Collection)
    (prism.internal HttpResponseDelegate)))

(defprotocol ^:private ->Array
  (->array ^"[Ljava.lang.String;" [_]))

(extend-protocol ->Array
  String
  (->array [this] (into-array String [this]))
  Collection
  (->array [^Collection this] (.toArray this))
  IPersistentCollection
  (->array [this] (into-array String this)))

(defprotocol ^:private HttpVersion
  (http-version [_]))

(extend-protocol HttpVersion
  Http1ClientResponse
  (http-version [_] :http-1.1)
  Http2ClientResponse
  (http-version [_] :http2))

(def http-methods
  {:get     Method/GET
   :post    Method/POST
   :put     Method/PUT
   :delete  Method/DELETE
   :head    Method/HEAD
   :options Method/OPTIONS
   :trace   Method/TRACE
   :patch   Method/PATCH
   :connect Method/CONNECT})

(defdelayed ^WebClient default-client (-> (WebClient/builder)
                                          .build))

(defn- add-headers! ^HttpClientRequest [^HttpClientRequest r m]
  (reduce-kv
    (fn add-header! [^ClientRequest r k v]
      (.header
        r
        (HeaderNames/create (name k))
        (->array (or v ""))))
    r
    m))

(defn- headers->map [headers]
  (persistent!
    (reduce
      (fn assoc-header! [acc ^Header header]
        (assoc! acc
                (-> (.name header) s/lower-case)
                (.get header)))
      (transient {})
      headers)))

(defn- execute-request! ^HttpResponseDelegate [^HttpClientRequest helidon-request {:keys [body] :as req}]
  (try
    (-> (if body
          (.submit helidon-request body)
          (.request helidon-request))
        HttpResponseDelegate.)
    (catch Exception e
      (throw (ex-info "Failed to execute request" req e)))))

(defn- req-uri ^URI
  "As per hato.client/ring-request->HttpRequest"
  [{:keys [scheme server-name server-port uri query-string]}]
  (-> (str (name scheme)
           "://"
           server-name
           (some->> server-port (str ":"))
           uri
           (some->> query-string (str "?")))
      URI.))

(defn- helidon-method ^Method [request-method]
  (or (get http-methods request-method)
      (some-> request-method name Method/create)
      Method/GET))

(defn request [{:keys [http-client request-method headers version timeout] :as req}]
  (let [^WebClient client (or http-client (default-client))
        resp (-> (.method client (helidon-method request-method))
                 (.uri (req-uri req))
                 (add-headers! headers)
                 (.protocolId (case version
                                :http2 "h2"
                                :http-1.1 "http/1.1"
                                nil))
                 (.readTimeout (Duration/ofMillis timeout))
                 (execute-request! req))]
    {:request      req
     :http-version (http-version (.delegate resp))
     :status       (.. resp status code)
     :body         (.inputStream resp)
     :headers      (-> (.headers resp)
                       headers->map)}))

(comment
  (request
    {:headers        {"test"            "test-val"
                      "user-agent"      "Prism local"
                      "accept-encoding" "gzip, deflate"}
     :server-port    nil
     :url            "https://webhook.site/f83b55ad-9f22-4ede-900b-7863556b1faf"
     :uri            "/f83b55ad-9f22-4ede-900b-7863556b1faf"
     :server-name    "webhook.site"
     :timeout        10000
     :query-string   "abc=a+test"
     :scheme         :https
     :request-method :get}))
