(ns prism.internal.http-client-helidon
  (:require
    [clojure.string :as s]
    [prism.core :refer [defdelayed]])
  (:import
    (clojure.lang IPersistentCollection)
    (io.helidon.common.uri UriEncoding)
    (io.helidon.http Header HeaderNames Method)
    (io.helidon.webclient.api ClientRequest HttpClientRequest WebClient)
    (java.time Duration)
    (java.util Collection)
    (prism.internal HttpResponseDelegate)))

(defprotocol ->Array
  (->array ^"[Ljava.lang.String;" [_]))

(extend-protocol ->Array
  String
  (->array [this] (into-array String [this]))
  Collection
  (->array [^Collection this] (.toArray this))
  IPersistentCollection
  (->array [this] (into-array String this)))

(def http-methods
  {:get     Method/GET
   :post    Method/POST
   :put     Method/PUT
   :delete  Method/DELETE
   :head    Method/HEAD
   :options Method/OPTIONS
   :trace   Method/TRACE
   :patch   Method/PATCH
   :connect Method/CONNECT})

(defdelayed ^WebClient default-client (-> (WebClient/builder)
                                          .build))

(defn- add-headers! ^HttpClientRequest [^HttpClientRequest r m]
  (reduce-kv
    (fn add-header! [^ClientRequest r k v]
      (.header
        r
        (HeaderNames/create (name k))
        (->array (or v ""))))
    r
    m))

(defn- headers->map [headers]
  (persistent!
    (reduce
      (fn assoc-header! [acc ^Header header]
        (assoc! acc
                (-> (.name header) s/lower-case)
                (.get header)))
      (transient {})
      headers)))

(defn request [{:keys [url body timeout request-method headers query-string http-client] :as req}]
  (let [method (or (get http-methods request-method)
                   (some-> request-method name Method/create)
                   Method/GET)
        uri-string (cond-> (UriEncoding/encodeUri url)
                           (seq query-string) (str \? query-string))
        ^WebClient client (or http-client (default-client))
        helidon-request (-> (.method client method)
                            (.skipUriEncoding true)
                            (.uri ^String uri-string)
                            (add-headers! headers)
                            (.readTimeout (Duration/ofMillis timeout)))
        resp (-> (if body
                   (.submit helidon-request body)
                   (.request helidon-request))
                 HttpResponseDelegate.)]
    {:request req
     :status  (.. resp status code)
     :body    (.inputStream resp)
     :headers (-> (.headers resp)
                  headers->map)}))
