(ns prism.ses
  (:require
    [hato.middleware :as hmw]
    [prism.aws :as aws]
    [prism.core :as prism :refer [defdelayed]]
    [taoensso.timbre :as log]))

(defdelayed ^:private ses-sender-address (-> (prism/config) :ses :sender-address))

(defdelayed ^:private ses-client
  (let [{{:keys [region] :as ses-config} :ses} (prism/config)]
    (->> (assoc ses-config
           :service "ses"
           :url (str "https://email"
                     (when region \.)
                     region
                     ".amazonaws.com"))
         (aws/create-client ::aws/static-url))))

(defn- ses-request-body [{:keys [address subject body]}]
  {:Content          {:Simple {:Body    {:Html {:Data body}}
                               :Subject {:Data subject}}}
   :Destination      {:ToAddresses [address]}
   :FromEmailAddress (ses-sender-address)})

(defn send-email! [email-details & {:as http-config}]
  (let [{:keys [status]
         :as   response} (aws/request
                           (ses-client) "/v2/email/outbound-emails"
                           (assoc http-config
                             :method :post
                             :json (ses-request-body email-details)))]
    (when-not (hmw/unexceptional-status? status)
      (log/with-context
        {:details (dissoc email-details :body)
         :body    (:body response)
         :status  status}
        (log/info "Failed to send email.")))
    response))

(comment
  (send-email! "test@email.com"
               "prism test subject"
               "prism test body"))
