(ns prism.greptime
  (:require
    [clojure.string :as s]
    [honey.sql :as hsql]
    [honey.sql.protocols :as hsp]
    [prism.core :refer [defdelayed] :as prism]
    [prism.ilp :as ilp]
    [prism.internal.classpath :as cp]
    [prism.json :as json]
    [prism.services :as services])
  (:import
    (clojure.lang IPersistentList IPersistentSet IPersistentVector ISeq Sequential)
    (java.time Instant)))

(defn- format-with [_ withs]
  (let [{:keys [clause params]} (reduce-kv
                                  (fn [acc k v]
                                    (let [[sql param] (hsql/format-expr v)
                                          k (hsql/format-entity k)]
                                      (-> (update acc :clause str k "=" sql ",")
                                          (cond-> (some? param) (update :params conj param)))))
                                  {:clause "with("
                                   :params []}
                                  withs)
        clause (-> (subs clause 0 (dec (count clause)))
                   (str ")"))]
    (into [clause] params)))

(defn- format-engine [_ engine]
  [(str "engine=" engine)])

(defn- sqlize-coll [coll]
  (as-> (mapv hsp/sqlize coll) $
        (s/join ", " $)
        (str \( $ \))))

(extend-protocol hsp/InlineValue
  Sequential
  (sqlize [this] (sqlize-coll this))
  IPersistentVector
  (sqlize [this] (sqlize-coll this))
  IPersistentList
  (sqlize [this] (sqlize-coll this))
  IPersistentSet
  (sqlize [this] (sqlize-coll this))
  ISeq
  (sqlize [this] (sqlize-coll this)))

(defn format-interval [_ interval]
  [(str "INTERVAL(" (hsql/format-entity interval) ")")])

(defn format-set-ttl [_ ttl]
  [(str "SET 'ttl'='" ttl \')])

(hsql/register-clause! :insert-into :insert-into :select)
(hsql/register-clause! :with-metadata format-with nil)
(hsql/register-clause! :engine format-engine :with-metadata)
(hsql/register-clause! :tags :values :join-by)
(hsql/register-clause! :interval format-interval :order-by)
(hsql/register-clause! :set-ttl format-set-ttl :add-column)
(hsql/register-clause! :as-select :select :create-table)
(hsql/register-clause! :sink-to :insert-into :as-select)
(hsql/register-clause! :create-flow :insert-into :sink-to)

(comment
  (hsql/format
    {:alter-table :x :set-ttl "1d"})
  (hsql/format
    {:create-table :my-table
     :with-columns [[:ts [:timestamp 9]]
                    [:tag-one :string]
                    [:tag-two :string]
                    [[:time-index :ts]]
                    [[:primary-key :tag-one :tag-two]]]})
  (hsql/format
    {:create-flow :new-flow
     :sink-to     :my-table
     :as-select   :a
     :from        :other-table
     :where       [:= :a 123]
     :group-by    [:b]}
    {}))

(defdelayed ^:private default-db (-> (prism/config) :greptime :dbname))

(defn- query-params [db] {:db db})

(defn- handle-http-result [response]
  (when (and (instance? Exception response)
             (some-> (ex-data response)
                     :body
                     (s/includes? "read only")))
    (services/invalidate-service! :greptime))
  response)

(defn- greptime-sql-request! [^String sql db]
  (-> (services/service-request
        :greptime "/v1/sql"
        {:method            :post
         :query-params      (query-params db)
         :form-params       {:sql sql}
         :catch-exceptions? true
         :retry             true
         :as                :json})
      handle-http-result))

(defn- ms->inst ^Instant [ms]
  (Instant/ofEpochMilli ms))

(defn- nanos->inst ^Instant [ns]
  (Instant/ofEpochSecond 0 ns))

(defn- build-format-fn* [column-schemas]
  (let [columns (mapv
                  (fn [{:keys [name data_type]}]
                    {:name   (keyword (prism/snake->kebab name))
                     :parser (case data_type
                               "TimestampMillisecond" ms->inst
                               "TimestampNanosecond" nanos->inst
                               identity)})
                  column-schemas)]
    (fn format-row [row]
      (->> (mapv vector row columns)
           (reduce (fn [r [row-column metadata]]
                     (assoc! r
                             (:name metadata)
                             ((:parser metadata) row-column)))
                   (transient {}))
           persistent!))))

(def ^:private build-format-fn (cp/if-ns 'clojure.core.memoize
                                 (clojure.core.memoize/lru build-format-fn* :lru/threshold 20)
                                 (memoize build-format-fn*)))

(defn- format-records [{:keys [schema rows]}]
  (let [format-fn (-> (:column_schemas schema)
                      build-format-fn)]
    (mapv format-fn rows)))

(comment (format-records
           {:schema {:column_schemas [{:name "field_name", :data_type "String"}
                                      {:name "greptime_value", :data_type "Float64"}
                                      {:name "greptime_timestamp", :data_type "TimestampMillisecond"}]}
            :rows   [["apps" 0.01882346944538697 1706138100000]]}))

(defn- format-success-response [{:keys [output]}]
  (mapv (fn [{:keys [records] :as res}]
          (if (nil? records)
            res
            (format-records records)))
        output))

(defn- handle-sql-response [{:keys [code] :as response} cause]
  (case (long (or code 0))
    0 (format-success-response response)
    4001 nil ;table doesn't exist - https://github.com/GreptimeTeam/greptimedb/blob/3029b47a89ca55c422a268d2d0e2ba38fbff9e95/src/common/error/src/status_code.rs
    (throw (ex-info "Error executing Greptime query" response cause))))

(defn- try-read-json [body]
  (prism/try-or (json/json->clj body) nil))

(defn- execute-sql! [sql db]
  (let [response (greptime-sql-request! sql db)]
    (if-not (instance? Exception response)
      (-> response
          :body
          (handle-sql-response nil))
      (if-let [body (-> response
                        ex-data
                        :body
                        try-read-json)]
        (handle-sql-response body response)
        (throw response)))))

(defn execute!
  ([query params] (execute! query params (default-db)))
  ([query params db]
   (-> (hsql/format query {:inline true
                           :params params})
       prism/vec-first
       (execute-sql! db))))

(defn execute-batch!
  ([queries] (execute-batch! queries (default-db)))
  ([queries db]
   (-> (eduction (map (fn [[query params]]
                        (-> (hsql/format query {:inline true
                                                :params params})
                            prism/vec-first)))
                 queries)
       (->> (s/join ";"))
       (execute-sql! db))))

(defn insert!
  ([points] (insert! points (default-db)))
  ([points db]
   (when-let [body (ilp/points->string points)]
     (-> (services/service-request
           :greptime "/v1/influxdb/write"
           {:method            :post
            :content-type      :application/x-www-form-urlencoded
            :query-params      (assoc (query-params db) "precision" "ns")
            :catch-exceptions? true
            :retry             true
            :body              body})
         handle-http-result))))

(comment
  (execute! [:raw "select * from uc_metrics where ts < ''"]
            {})
  (execute! [:raw "select * from not_a_table"]
            {}))
