(ns prism.greptime
  (:require
    [clojure.string :as s]
    [honey.sql :as hsql]
    [honey.sql.protocols :as hsp]
    [prism.core :refer [defdelayed] :as prism]
    [prism.ilp :as ilp]
    [prism.internal.classpath :as cp]
    [prism.services :as services])
  (:import
    (clojure.lang IPersistentList IPersistentSet IPersistentVector ISeq Sequential)
    (java.net URLEncoder)
    (java.nio.charset StandardCharsets)
    (java.time Instant)))

(defn- format-with [_ withs]
  (let [{:keys [clause params]} (reduce-kv
                                  (fn [acc k v]
                                    (let [[sql param] (hsql/format-expr v)
                                          k (hsql/format-entity k)]
                                      (-> (update acc :clause str k "=" sql ",")
                                          (cond-> (some? param) (update :params conj param)))))
                                  {:clause "with("
                                   :params []}
                                  withs)
        clause (-> (subs clause 0 (dec (count clause)))
                   (str ")"))]
    (into [clause] params)))

(defn- format-engine [_ engine]
  [(str "engine=" engine)])

(defn- sqlize-coll [coll]
  (as-> (mapv hsp/sqlize coll) $
        (s/join ", " $)
        (str \( $ \))))

(extend-protocol hsp/InlineValue
  Sequential
  (sqlize [this] (sqlize-coll this))
  IPersistentVector
  (sqlize [this] (sqlize-coll this))
  IPersistentList
  (sqlize [this] (sqlize-coll this))
  IPersistentSet
  (sqlize [this] (sqlize-coll this))
  ISeq
  (sqlize [this] (sqlize-coll this)))

(defn format-interval [_ interval]
  [(str "INTERVAL(" (hsql/format-entity interval) ")")])

(hsql/register-clause! :insert-into :insert-into :select)
(hsql/register-clause! :with-metadata format-with nil)
(hsql/register-clause! :engine format-engine :with-metadata)
(hsql/register-clause! :tags :values :join-by)
(hsql/register-clause! :interval format-interval :order-by)

(defdelayed ^:private query-params {:db (-> (prism/config) :greptime :dbname)})

(defn- handle-exception-response [e]
  (when (-> (ex-data e)
            :body
            (s/includes? "read only"))
    (services/invalidate-service! :greptime))
  (throw e))

(defn- greptime-sql-request! [^String sql]
  (-> (services/service-request
        :greptime "/v1/sql"
        {:method            :post
         :content-type      :application/x-www-form-urlencoded
         :query-params      (query-params)
         :body              (str "sql=" (URLEncoder/encode sql StandardCharsets/UTF_8))
         :catch-exceptions? true
         :retry             true
         :as                :json})
      handle-exception-response))

(defn- ms->inst ^Instant [ms]
  (Instant/ofEpochMilli ms))

(defn- build-format-fn* [column-schemas]
  (let [columns (mapv
                  (fn [{:keys [name data_type]}]
                    {:name   (keyword (prism/snake->kebab name))
                     :parser (case data_type
                               "TimestampMillisecond" ms->inst
                               identity)})
                  column-schemas)]
    (fn format-row [row]
      (->> (mapv vector row columns)
           (reduce (fn [r [row-column metadata]]
                     (assoc! r
                             (:name metadata)
                             ((:parser metadata) row-column)))
                   (transient {}))
           persistent!))))

(def ^:private build-format-fn (cp/if-ns 'clojure.core.memoize
                                 (clojure.core.memoize/lru build-format-fn* :lru/threshold 20)
                                 (memoize build-format-fn*)))

(defn- format-records [{:keys [schema rows]}]
  (let [format-fn (-> (:column_schemas schema)
                      build-format-fn)]
    (mapv format-fn rows)))

(comment (format-records
           {:schema {:column_schemas [{:name "field_name", :data_type "String"}
                                      {:name "greptime_value", :data_type "Float64"}
                                      {:name "greptime_timestamp", :data_type "TimestampMillisecond"}]}
            :rows   [["apps" 0.01882346944538697 1706138100000]]}))

(defn- format-success-response [{:keys [output]}]
  (mapv (fn [{:keys [records] :as res}]
          (if (nil? records)
            res
            (format-records records)))
        output))

(defn- handle-response [{:keys [code] :as response}]
  (case (long (or code 0))
    0 (format-success-response response)
    3000 nil ;table doesn't exist
    (throw (ex-info "Error executing Greptime query" response))))

(defn- execute-sql! [sql]
  (-> (greptime-sql-request! sql)
      :body
      handle-response))

(defn execute! [query params]
  (-> (hsql/format query {:inline true
                          :params params})
      prism/vec-first
      execute-sql!))

(defn execute-batch! [queries]
  (->> (eduction (map (fn [[query params]]
                        (-> (hsql/format query {:inline true
                                                :params params})
                            prism/vec-first)))
                 queries)
       (s/join ";")
       execute-sql!))

(defn insert! [points]
  (when-let [body (ilp/points->string points)]
    (-> (services/service-request
          :greptime "/v1/influxdb/write"
          {:method            :post
           :content-type      :application/x-www-form-urlencoded
           :query-params      (assoc (query-params) "precision" "ns")
           :catch-exceptions? true
           :retry             true
           :body              body})
        handle-exception-response)))
