(ns prism.http-server
  (:require
    [malli.core :as m]
    [malli.transform :as mt]
    [malli.util :as mu]
    [muuntaja.core :as muuntaja]
    [muuntaja.interceptor :as mi]
    [prism.core :as prism]
    [prism.internal.classpath :as cp]
    [reitit.coercion.malli :as mc]
    [reitit.core :as r]
    [reitit.http :as rhttp]
    [reitit.http.coercion :as rhc]
    [reitit.http.interceptors.exception :as exception]
    [reitit.http.interceptors.parameters :as parameters]
    [reitit.impl :as ri]
    [reitit.interceptor.sieppari :as sieppari]
    [reitit.ring :as ring]
    [reitit.spec :as rs]
    [ring.adapter.jetty9 :as jetty])
  (:import
    (java.time Instant)
    (org.eclipse.jetty.server Server)))

(def muuntaja-instance (cp/if-ns 'prism.json
                         (-> (update muuntaja/default-options :formats assoc "application/json" prism.json/muuntaja-format)
                             muuntaja/create)
                         muuntaja/instance))

(defn- reference-expand [endpoints-registry]
  (fn [data opts]
    (if (keyword? data)
      (some-> data
              endpoints-registry
              (r/expand opts)
              (assoc :name data))
      (r/expand data opts))))

(defn- exception-handler [message exception request]
  {:status 500
   :body   {:message   message
            :exception (str exception)
            :uri       (:uri request)}})

(defn- exception-wrapper [handler e request]
  (handler e request))

(cp/when-ns 'taoensso.timbre
  (defn- exception-wrapper [handler e request]
    (let [context (-> (select-keys request [:request-method :scheme :content-length :content-type :uri
                                            :parameters :query-params :path-params :query-string :body-params])
                      (update-in [:parameters :header] dissoc :authorization))
          {ex-type :type
           :as     ex-data} (ex-data e)]
      (case ex-type
        :reitit.coercion/request-coercion (taoensso.timbre/with-context
                                            context
                                            (taoensso.timbre/debugf "Request coercion error in '%s'." (:in ex-data)))
        :reitit.coercion/response-coercion (taoensso.timbre/with-context
                                             (assoc context :response (:response ex-data))
                                             (taoensso.timbre/error "Response coercion error."))
        (:ring.util.http-response/response ::ring/response) nil
        (taoensso.timbre/with-context
          context
          (taoensso.timbre/errorf e "Unexpected exception while handling HTTP request."))))
    (handler e request)))

(defn exception-interceptor []
  (-> (merge
        exception/default-handlers
        {::error                           (partial exception-handler "error")
         ::exception                       (partial exception-handler "exception")
         :ring.util.http-response/response exception/http-response-handler
         ::exception/default               (partial exception-handler "default")
         ::exception/wrap                  exception-wrapper})
      exception/exception-interceptor))

(defn default-interceptors [muuntaja]
  [(mi/format-interceptor muuntaja)
   (exception-interceptor)
   (rhc/coerce-response-interceptor)
   (parameters/parameters-interceptor)
   (rhc/coerce-request-interceptor)])

(defn- try-parse-inst [s]
  (if (string? s)
    (try
      (Instant/parse ^String s)
      (catch Exception _
        s))
    s))

(defn -json-transformer [strip-extra-keys?]
  (mt/transformer
    {:name     :inst
     :decoders {'inst? try-parse-inst}
     :encoders {'inst? mt/-any->string}}
    (when strip-extra-keys?
      (-> (mt/strip-extra-keys-transformer)
          m/-transformer-chain
          first
          (update-vals #(dissoc % :map-of))))
    mt/default-value-transformer
    mt/json-transformer))

(def string-transformer
  (mt/transformer
    {:name     :inst
     :decoders {'inst? try-parse-inst}
     :encoders {'inst? mt/-any->string}}
    mt/string-transformer
    mt/default-value-transformer))

(def default-coerce-options
  (let [json-transformer (-json-transformer false)]
    {:transformers {:body     {:default json-transformer
                               :formats {"application/json" json-transformer}}
                    :string   {:default string-transformer}
                    :response {:default json-transformer}}
     :compile      mu/open-schema}))

(defn- attach-path-params [expand-map path-params routes-data]
  (let [expand-map (atom expand-map)
        routes-data (mapv (fn attach-path-parameters
                            ([d] (attach-path-parameters [] d))
                            ([parent-path-params original]
                             (let [path (prism/vec-first original)
                                   data (subvec original 1)
                                   first-data (prism/vec-first data)
                                   path-params (->> (ri/parse path nil)
                                                    :path-params
                                                    (into parent-path-params
                                                          (map (fn [kw] [kw (get path-params kw string?)]))))
                                   attach-method-path-params-fn (fn [data] (update-in
                                                                             data
                                                                             [:parameters :path]
                                                                             (fnil identity (into [:map] path-params))))]
                               (cond
                                 (vector? first-data) (into [path] (mapv #(attach-path-parameters path-params %) data))
                                 (seq path-params) (cond
                                                     (fn? first-data) {:handler    first-data
                                                                       :parameters {:path (into [:map] path-params)}}
                                                     (:handler first-data) (assoc first-data
                                                                             :parameters {:path (into [:map] path-params)})
                                                     (map? first-data) [path (update-vals
                                                                               first-data
                                                                               #(if (keyword? %)
                                                                                  (do (swap!
                                                                                        expand-map
                                                                                        update
                                                                                        %
                                                                                        attach-method-path-params-fn)
                                                                                      %)
                                                                                  (attach-method-path-params-fn %)))])
                                 :else original))))
                          routes-data)]
    {:expand-map  @expand-map
     :routes-data routes-data}))

(defn create-handler [routes-data & {:keys [expand expand-map coerce-options
                                            interceptors muuntaja
                                            inject-match? inject-router?
                                            attach-path-params? path-params
                                            default-options-endpoint
                                            compile]
                                     :or   {muuntaja            muuntaja-instance
                                            attach-path-params? true
                                            inject-match?       false
                                            inject-router?      false}}]
  (let [{:keys [expand-map routes-data]} (if attach-path-params?
                                           (attach-path-params expand-map path-params routes-data)
                                           (prism/kw-map expand-map routes-data))
        expand (or expand
                   (when (seq expand-map)
                     (reference-expand expand-map))
                   r/expand)
        interceptors (or interceptors
                         (default-interceptors muuntaja))
        coercion (-> (merge default-coerce-options coerce-options)
                     mc/create)]
    (rhttp/ring-handler
      (rhttp/router
        routes-data
        (cond-> {:expand   expand
                 :validate rs/validate
                 :data     {:coercion coercion}}
                default-options-endpoint (assoc ::rhttp/default-options-endpoint default-options-endpoint)
                compile (assoc :compile compile)))

      (ring/routes
        (ring/create-default-handler))

      {:executor       sieppari/executor
       :interceptors   interceptors
       :inject-match?  inject-match?
       :inject-router? inject-router?})))

(defn- configure-server [^Server server]
  (.addShutdownHook (Runtime/getRuntime)
                    (Thread. #(do
                                (cp/when-ns 'taoensso.timbre
                                  (taoensso.timbre/info "Stopping server"))
                                (.stop server)
                                (cp/when-ns 'prism.postgres
                                  (cp/when-ns 'taoensso.timbre
                                    (taoensso.timbre/info "Shutting down DB connection pool"))
                                  (.close (prism.postgres/data-src)))))))

(defn start-server! ^Server [handler]
  (let [port (-> (prism/config) :port)]
    (cp/when-ns 'taoensso.timbre
      (taoensso.timbre/infof "Starting server on port %d" port))
    (jetty/run-jetty handler {:port             port
                              :h2c?             true
                              :join?            false
                              :virtual-threads? true
                              :configurator     configure-server})))

(comment
  (do (require '[reitit.openapi :as openapi])
      (require '[prism.http :as http])
      (require '[clojure.java.io :as io]))
  (->> (create-handler
         [["/openapi.json"
           {:get {:handler        (openapi/create-openapi-handler)
                  :openapi        {:info {:title "my nice api" :version "0.0.1"}}
                  :inject-router? true
                  :no-doc         true}}]
          ["/test/:abc"
           ["" {:get  ::test
                :post ::test}]
           ["/body" {:get {:handler    (fn [req]
                                         (pr req)
                                         (println)
                                         {:status 201
                                          :body   {:with "body"}})
                           :parameters {:body [:map [:a int?]]
                                        :path map?}}}]
           ["/error" {:handler    (fn [req]
                                    [req]
                                    (pr req)
                                    (println)
                                    (throw (ex-info "short and stout" {:status 418})))
                      :parameters {:body [:map
                                          [:jay [:enum "son" "daughter"]]]}}]
           ["/error2" (fn [req]
                        [req]
                        (pr req)
                        (println)
                        (throw (ex-info "short and stout" {:status 418})))]]]
         :expand-map {::test {:handler (fn [req]
                                         (pr req)
                                         (println)
                                         {:status 200})}}
         :interceptors (conj (default-interceptors muuntaja-instance) openapi/openapi-feature))
       start-server!
       (def server))
  (.stop server)
  (->> (http/request {:method :get
                      :url    (str "http://localhost:" (-> (prism/config) :port) "/openapi.json")})
       :body
       (spit (io/file "dev/openapi.json")))
  (http/request {:method :get
                 :url    (str "http://localhost:" (-> (prism/config) :port) "/test/value")})
  (http/request {:method           :get
                 :throw-exceptions false
                 :as               :auto
                 :json             {:a 1 :b 2}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/123/body")})
  (http/request {:method           :put
                 :timeout          100000
                 :as               :auto
                 :throw-exceptions false
                 :json             {:jay "x"}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/value/error")}))
