(ns prism.json
  (:require
    [charred.api :as charred]
    [clojure.stacktrace :as stacktrace]
    [clojure.string :as s]
    [prism.core :as prism]
    [prism.internal.classpath :as cp])
  (:import
    (java.io StringWriter Writer)))

(def default-write-config {:escape-unicode       false
                           :escape-slash         false
                           :escape-js-separators false})
(def default-read-config {:key-fn keyword})

(def json->clj (charred/parse-json-fn default-read-config))

(def write-json (charred/write-json-fn default-write-config))

(defn write-json-string [o]
  (with-open [sw (StringWriter.)]
    (write-json sw o)
    (.toString sw)))

(defn json->clj-checked [s]
  (when-not (s/blank? s)
    (json->clj s)))

(def configure-logging! (cp/missing-dep "timbre" "aero"))
(cp/when-ns 'taoensso.timbre
  (cp/when-ns 'aero.core
    (let [structure-log (fn structure-log [{:keys [msg_ ?err ?ns-str ?line level timestamp_]}]
                          (cond-> {:caller (str ?ns-str \: ?line)
                                   :level  (name level)
                                   :thread (.getName (Thread/currentThread))
                                   :msg    (force msg_)
                                   :ts     (force timestamp_)}
                                  ?err (assoc :stacktrace (with-out-str
                                                            (stacktrace/print-cause-trace ?err)))
                                  (seq (ex-data ?err)) (assoc :ex-data (write-json-string (ex-data ?err)))))
          json-appender {:enabled?   true
                         :async?     false
                         :min-level  nil
                         :rate-limit nil
                         :output-fn  :inherit
                         :fn         (fn json-appender-fn [{:keys [output_]}]
                                       (.write ^Writer *err*
                                               (str
                                                 (write-json-string (force output_))
                                                 taoensso.encore/system-newline))
                                       (.flush ^Writer *err*))}]
      (defn configure-logging! []
        (taoensso.timbre/set-config!
          {:min-level      (-> (prism/config) :log-level keyword)
           :ns-filter      #{"*"}
           :middleware     []
           :timestamp-opts taoensso.timbre/default-timestamp-opts
           :output-fn      structure-log
           :appenders      {:println json-appender}})))))

(declare muuntaja-format)
(cp/when-ns 'muuntaja.format.charred
  (def muuntaja-format
    {:name    "application/json"
     :decoder [muuntaja.format.charred/decoder default-read-config]
     :encoder [muuntaja.format.charred/encoder default-write-config]}))
