(ns prism.greptime
  (:require
    [honey.sql :as hsql]
    [prism.core :refer [defdelayed] :as prism]
    [prism.http :as http]
    [prism.services :as services]
    [prism.ilp :as ilp])
  (:import
    (java.net URLEncoder)
    (java.nio.charset StandardCharsets)
    (java.time Instant)))

(hsql/register-clause!
  :insert-into :insert-into :select)

(defdelayed ^:private query-params {:db (-> (prism/config) :greptime :dbname)})

(defn- greptime-sql-request! [^String sql]
  (services/service-request
    :greptime
    "/v1/sql"
    {:content-type :application/x-www-form-urlencoded
     :query-params (query-params)
     :body         (str "sql=" (URLEncoder/encode sql StandardCharsets/UTF_8))
     :method       :post
     :retry        true
     :as           :json}))

(defn- ms->inst ^Instant [ms]
  (Instant/ofEpochMilli ms))

(defn- build-format-fn* [column-schemas]
  (let [columns (mapv
                  (fn [{:keys [name data_type]}]
                    {:name   (keyword (prism/snake->kebab name))
                     :parser (case data_type
                               "TimestampMillisecond" ms->inst
                               identity)})
                  column-schemas)]
    (fn format-row [row]
      (->> (mapv vector row columns)
           (reduce (fn [r [row-column metadata]]
                     (assoc! r
                             (:name metadata)
                             ((:parser metadata) row-column)))
                   (transient {}))
           persistent!))))

(def ^:private build-format-fn (memoize build-format-fn*))

(defn- format-records [{:keys [schema rows]}]
  (let [format-fn (-> (:column_schemas schema)
                      build-format-fn)]
    (mapv format-fn rows)))

(comment (format-records
           {:schema {:column_schemas [{:name "field_name", :data_type "String"}
                                      {:name "greptime_value", :data_type "Float64"}
                                      {:name "greptime_timestamp", :data_type "TimestampMillisecond"}]}
            :rows   [["apps" 0.01882346944538697 1706138100000]]}))

(defn- format-success-response [{:keys [output]}]
  (mapv (fn [{:keys [records] :as res}]
          (if (nil? records)
            res
            (format-records records)))
        output))

(defn- handle-response [{:keys [code] :as response}]
  (case (long (or code 0))
    0 (format-success-response response)
    3000 nil ;table doesn't exist
    (throw (ex-info "Error executing Greptime query" response))))

(defn execute! [query params]
  (-> (hsql/format query {:inline true
                          :params params})
      prism/vec-first
      greptime-sql-request!
      :body
      handle-response))

(defn insert! [points]
  (when-let [body (ilp/points->string points)]
    (services/service-request
      :greptime
      "/v1/influxdb/write"
      {:content-type :application/x-www-form-urlencoded
       :query-params (assoc (query-params) "precision" "ns")
       :method       :post
       :body         body
       :retry        true})))
