(ns prism.minio
  (:require
    [clojure.java.io :as io]
    [hato.middleware :as hmw]
    [prism.core :refer [defdelayed] :as prism]
    [prism.internal.classpath :as cp]
    [prism.services :as services])
  (:import
    (com.github.davidmoten.aws.lw.client BaseUrlFactory Client HttpMethod Request)
    (java.net ConnectException)
    (org.json XML)))

(def ^:private dns-url-factory
  (reify
    BaseUrlFactory
    (create [_ _ _]
      (services/service-url! :minio))))

(defdelayed ^:private ^Client s3
  (let [{{:keys [username password]} :minio} (prism/config)]
    (-> (Client/s3)
        (.region "us-east-1")
        (.accessKey username)
        (.secretKey password)
        (.baseUrlFactory dns-url-factory)
        .build)))

(defn- parse-response-stream [input-stream]
  (->> input-stream io/reader XML/toJSONObject .toMap))

(defn- with-connection-handling [f]
  (try
    (f)
    (catch ConnectException _
      (services/invalidate-service! :minio)
      (when-not (services/service-url! :minio)
        (throw (ex-info "Could not resolve domain name for minio" {:status 503}))))))

(defn- make-request! [^Request request]
  (if-let [response (with-connection-handling #(.responseInputStream request))]
    (parse-response-stream response)
    (recur request)))

(defn list-objects [{:keys [bucket delimiter prefix]}]
  (-> (.path (s3) (into-array String [bucket]))
      (.query "list-type" "2")
      (.query "delimiter" delimiter)
      (.query "prefix" prefix)
      make-request!
      (get "ListBucketResult")))

(defn delete-object! [{:keys [bucket key]}]
  (-> (.path (s3) (into-array String [bucket key]))
      (.method HttpMethod/DELETE)
      make-request!))

(defn put-object! [{:keys [bucket key body]}]
  (-> (.path (s3) (into-array String [bucket key]))
      (.requestBody ^String body)
      (.method HttpMethod/PUT)
      make-request!))

(defn- stream-response [^Request req]
  (let [stream (.responseInputStream req)]
    (if (-> (.statusCode stream)
            hmw/unexceptional-status?)
      stream
      (cp/when-ns 'taoensso.timbre
        (let [error (-> (parse-response-stream stream)
                        (get "Error"))]
          (when-not (->> (get error "Code")
                         (= "NoSuchKey"))
            (taoensso.timbre/warnf "Received errored minio response: '%s'" error)))))))

(defn get-object [{:keys [bucket key]}]
  (let [req (.path (s3) (into-array String [bucket key]))]
    (with-connection-handling
      #(stream-response req))))

(comment
  (get-object {:bucket "ffwd-temp-test"
               :key    "some-key"})
  (put-object! {:bucket "ffwd-temp-test"
                :key    "some-key"
                :body   (slurp (io/file "deps.edn"))})
  (list-objects {:prefix    "rules/"
                 :bucket    "ffwd-loki"
                 :delimiter "/"}))

