(ns prism.postgres
  (:require
    [clojure.string :as s]
    [honey.sql :as hsql]
    [next.jdbc :as jdbc]
    [next.jdbc.connection :as connection]
    [next.jdbc.date-time :as date-time]
    [next.jdbc.prepare :refer [SettableParameter]]
    [next.jdbc.result-set :refer [ReadableColumn as-unqualified-kebab-maps]]
    [prism.core :refer [defdelayed] :as prism]
    [prism.internal.classpath :as cp])
  (:import
    (clojure.lang Associative Counted ILookup IObj IPersistentCollection IPersistentMap IPersistentVector Seqable)
    (com.zaxxer.hikari HikariDataSource)
    (java.sql Connection PreparedStatement Timestamp)
    (java.time Instant)
    (java.util Date UUID)
    (org.postgresql.util PGobject)))

(cp/eval-when (cp/checked-require '[prism.json :as json])
  (defn- ->pgobject
    "Transforms Clojure data to a PGobject that contains the data as
    JSON. PGObject type defaults to `jsonb` but can be changed via
    metadata key `:pgtype`"
    [x]
    (let [pg-type (or (:pg-type (meta x)) "jsonb")]
      (doto (PGobject.)
        (.setType pg-type)
        (.setValue (json/write-json-string x)))))

  (defn- <-pgobject
    "Transform PGobject containing `json` or `jsonb` value to Clojure
    data."
    [^PGobject v]
    (let [type (.getType v)
          value (.getValue v)]
      (if (#{"jsonb" "json"} type)
        (let [v (json/json->clj value)]
          (cond-> v
                  (instance? IObj v) (with-meta {:pg-type type})))
        value)))

  (extend-protocol SettableParameter
    IPersistentMap
    (set-parameter [m ^PreparedStatement s i]
      (.setObject s i (->pgobject m)))

    IPersistentVector
    (set-parameter [v ^PreparedStatement s i]
      (.setObject s i (->pgobject v))))

  (extend-protocol ReadableColumn
    PGobject
    (read-column-by-label [^PGobject v _]
      (<-pgobject v))
    (read-column-by-index [^PGobject v _ _]
      (<-pgobject v))))

(extend-protocol SettableParameter
  Date
  (set-parameter [v ^PreparedStatement s i]
    (.setTimestamp s i (-> v
                           (inst-ms)
                           (Timestamp.))))

  Instant
  (set-parameter [v ^PreparedStatement s i]
    (.setTimestamp s i (Timestamp/from v))))

(extend-protocol ReadableColumn
  UUID
  (read-column-by-label [^PGobject v _]
    (str v))
  (read-column-by-index [^PGobject v _ _]
    (str v)))

(defn- json-get-str [_ [& exprs]]
  (let [[sqls params] (hsql/format-expr-list exprs)
        sql (-> (repeat (- (count sqls) 2) " -> ")
                vec
                (into [" ->> " nil])
                (->> (interleave sqls))
                s/join)]
    (into [sql] params)))

(date-time/read-as-instant)
(hsql/register-fn! :->> json-get-str)
(hsql/register-op! :-> :variadic true)
(hsql/register-op! :||)

(defdelayed ^HikariDataSource data-src
  (let [{:keys [max-connections]
         :as   postgres-config} (-> (prism/config)
                                    :postgres)]
    (connection/->pool HikariDataSource
                       {:jdbcUrl                   (-> (assoc postgres-config :dbtype "postgres")
                                                       connection/jdbc-url)
                        :initializationFailTimeout 0
                        :maximumPoolSize           (or max-connections 10)})))

(declare ^:dynamic ^Connection *c*)
(declare ^:dynamic ^Connection *txn*)

(defn connect ^Connection [] (jdbc/get-connection (data-src)))

(defmacro with-conn [& body]
  `(with-open [c# (connect)]
     (binding [*c* c#]
       ~@body)))

(defmacro with-txn [& body]
  `(jdbc/with-transaction
     [~'txn (data-src)]
     (binding [*txn* ~'txn
               *c* ~'txn]
       ~@body)))

(defn- format-sql [statement params]
  (hsql/format statement {:params params}))

(declare ^:private query-chache)
(cp/eval-when (and (cp/checked-require '[cloffeine.cache :as c])
                   (cp/checked-require '[clojure.core.cache :as ccc]))
  (deftype CloffeineCache [cache]
    ccc/CacheProtocol
    (lookup [_ k] (c/get-if-present cache k))
    (lookup [_ k not-found] (or (c/get-if-present cache k) not-found))
    (has? [_ e] (some? (c/get-if-present cache e)))
    (hit [this _] this)
    (miss [this k v] (c/put! cache k v) this)
    (evict [this e] (c/invalidate! cache e) this)
    (seed [this _] this)

    Associative
    (assoc [this k v] (ccc/miss this k v))
    (entryAt [_ k] (find (c/as-map cache) k))
    (containsKey [this k] (ccc/has? this k))

    ILookup
    (valAt [this k] (ccc/lookup this k))
    (valAt [this k v] (ccc/lookup this k v))

    Seqable
    (seq [_] (seq (c/as-map cache)))

    IPersistentCollection
    (count [_] (count (c/as-map cache)))
    (cons [_ _] (throw (UnsupportedOperationException.)))
    (empty [this] (c/invalidate-all! cache) this)
    (equiv [this other] (identical? this other))

    Counted
    IPersistentMap
    (without [this k] (ccc/evict this k) this)

    Iterable
    (iterator [_] (.iterator ^Iterable (sequence (c/as-map cache)))))

  (def ^:private query-cache (atom (->CloffeineCache (c/make-cache))))

  (defn- format-sql [statement {cached? ::cached?
                                :as     options
                                :or     {cached? true}}]
    (let [params (dissoc options ::cached?)
          opts (cond-> {}
                       (seq params) (assoc :params params)
                       cached? (assoc :cache query-cache))]
      (hsql/format statement opts))))

(defn execute! [statement {:keys [::return-keys] :as opts}]
  (jdbc/execute! *c*
                 (format-sql statement opts)
                 (cond-> {:builder-fn as-unqualified-kebab-maps}
                         (seq return-keys) (assoc :return-keys (mapv (comp prism/vec-first hsql/format-expr) return-keys)))))

(defn batch-insert! [table params]
  (when (seq params)
    (let [params-template (-> (first params)
                              (update-vals (constantly 0)))
          ->ordered-params (apply juxt (keys params-template))
          statement (-> (format-sql {:insert-into table
                                     :values      [params-template]}
                                    {::cached? false})
                        (subvec 0 1))]
      (with-open [ps (jdbc/prepare *c* statement)]
        (->> (mapv ->ordered-params params)
             (jdbc/execute-batch! ps))))))

(defn exists? [exists-query params]
  (-> (execute! {:select [[true :exists]]
                 :where  [:exists (assoc exists-query :select [1])]}
                params)
      first
      :exists))

(defn updated? [result]
  (-> result prism/vec-first :next.jdbc/update-count (> 0)))

(defn rollback-txn! []
  (.rollback *txn*))
