# io.clj/logging

`logging` is a small Clojure library (really just one macro) designed
to wrap SLF4J's MDC feature. Using SLF4J and either
[Logback](http://logback.qos.ch/) or
[log4j](http://logging.apache.org/log4j/) it is possible to add some
(scoped) context (such as request-client ID etc.) to log messages as
key/value pairs.

Even though in Clojure we would use dynamic vars to have per-thread
information attached to log messages, MDC is directly supported by
some tools and gives a bit more flexibility.

## Usage

Add this to your project.clj:

```clojure
[io.clj/logging "0.8.1"]
```

After that, just `:use` (or `:require`) `io.clj.logging`:

```clojure
:use [io.clj.logging]
```

To wrap a function with some context:

```clojure
(with-logging-context {:source "somesource"
                       :target "differenttarget"}
  (error (Exception. "aaaahhhh") "some log message"))
```

This is roughly equivalent to this Java code:

```java
// backup the MDC context
MDC.put("source", "somesource");
MDC.put("target", "differenttarget");
try {
    log.error("some log message", new Exception("aaaahhhh"));
} finally {
    // restore the MDC context
}
```

`debug`, `info`, `warn`, and `error` are wrapped for your convenience and
delegate to `clojure.tools.logging`. Of course you can use
`clojure.tools.logging` as well for all your logging needs and only refer to
`with-logging-context` from this package if required. Just make sure to use
SLF4J in the backend.

### Convenience debugging function

When searching for errors it is often useful to plug some logging between
a function call and its parameters. Instead of forcing the user to extract
the parameters into their own vars, `logging` provides a `d` function to
log and return a single value (optionally with a context):

```clojure
(println (d "this will get logged and printed"))
```

or with context:

```clojure
(println (d {:user-id 123} "this will get logged with context and the string will get printed"))
```

### Nesting

The context can be nested without interfering with other scopes:

```clojure
(with-logging-context {:source "somesource"
                       :target "differenttarget"}
    (with-logging-context {:target "does not apply"}
        (error "Inner log message"))
    (error "Outer log message"))
```

In this case, `:source` stays the same, and `:target` gets a different value.

The log message `Inner log message` will have an MDC of `{:source
"somesource" :target "does not apply"}` whereas the `Outer log
message` will have an MDC of `{:source "somesource" :target
"differenttarget"}` again.

## Examples

Here are some examples how to configure SLF4J and Logback.

### Basic Example

In this example, we configure SLF4J (pulled in by `clj-logging`) and
`logback` as the backend. We also include a bridge from log4j to SLF4J
to have a single common logging backend to configure (logback).

```clojure
[io.clj/logging "0.8.1"]
[ch.qos.logback/logback-core "1.0.9"]
[ch.qos.logback/logback-classic "1.0.9"]
[org.slf4j/log4j-over-slf4j "1.7.2"]
```

The configuration for logback (stored in `logback.xml`) could look as follows.

```xml
<?xml version="1.0" encoding="UTF-8"?>

<configuration scan="true" scanPeriod="10 seconds">
    <appender name="STDOUT" class="ch.qos.logback.core.ConsoleAppender">
        <encoder>
            <pattern>%d{HH:mm:ss.SSS} %-5level [%thread] %logger - [%mdc] - %msg%n</pattern>
        </encoder>
    </appender>

    <root level="DEBUG">
        <appender-ref ref="STDOUT"/>
    </root>
</configuration>
```

Here we configure logback to watch for changes in the configuration
every 10 seconds and to log to standard out.

Here is an example log-line (as configured in the example directory),
here we put `client-ip` into the context.

```
13:49:44.813 INFO  [main] io.clj.logging.examples - [client-ip=192.168.1.1] - got a new workpackage
```

### JSON Logging

Logging in a machine readable format has some advantages over plain
text files. In this example, we'll configure logback to emit JSON
objects.

To use the JSON configuration below, add the following dependencies to your
project.clj:

```clojure
[io.clj/logging "0.8.1"]
[ch.qos.logback/logback-core "1.0.9"]
[ch.qos.logback/logback-classic "1.0.9"]
[org.slf4j/log4j-over-slf4j "1.7.2"]
[ch.qos.logback.contrib/logback-jackson "0.1.2"]
[com.fasterxml.jackson.core/jackson-databind "2.1.3"]
[ch.qos.logback.contrib/logback-json-classic "0.1.2"]
```

```xml
<?xml version="1.0" encoding="UTF-8"?>

<configuration scan="true" scanPeriod="10 seconds">
    <appender name="JSON" class="ch.qos.logback.core.ConsoleAppender">
        <encoder class="ch.qos.logback.core.encoder.LayoutWrappingEncoder">
            <layout class="ch.qos.logback.contrib.json.classic.JsonLayout">
                <jsonFormatter class="ch.qos.logback.contrib.jackson.JacksonJsonFormatter"/>
                <timestampFormat>yyyy-MM-dd'T'HH:mm:ss.SSS'Z'</timestampFormat>
                <timestampFormatTimezoneId>UTC</timestampFormatTimezoneId>
                <appendLineSeparator>true</appendLineSeparator>
            </layout>
        </encoder>
    </appender>

    <root level="DEBUG">
        <appender-ref ref="JSON"/>
    </root>
</configuration>
```

This configuration emits the following JSON object (compared to the previous example):

```json
{"timestamp":"2013-02-03T12:49:44.813Z","level":"INFO","thread":"main","mdc":{"client-ip":"192.168.1.1"},"logger":"io.clj.logging.examples","message":"got a new workpackage","context":"default"}
```

Or in a more readable form:
```json
{
    "context": "default",
    "level": "INFO",
    "logger": "io.clj.logging.examples",
    "mdc": {
        "client-ip": "192.168.1.1"
    },
    "message": "got a new workpackage",
    "thread": "main",
    "timestamp": "2013-02-03T12:49:44.813Z"
}
```

Note that the timestamp has been converted to UTC (from CET), and that
the `mdc` is available as nested JSON object.

### Example project

The `/example` folder contains a demo project with some more
configuration examples (like RollingFilaAppender).

## Thanks

The macro is based on a similar one in Malcom Sparks'
[clj-logging-config](https://github.com/malcolmsparks/clj-logging-config).

## License

Copyright © 2013-2014 Michael Jakl

Distributed under the Eclipse Public License, the same as Clojure.
