(ns io.chancerussell.aspy
  #?(:clj
      (:require
        [clojure.core.async :refer [chan pipe]]
        [clojure.core.async.impl.protocols :refer [ReadPort]]
        [clojure.pprint :as pp]
        [clojure.spec :as s])
     :cljs
     (:require
       [cljs.core.async :refer [chan pipe]]
       [cljs.core.async.impl.protocols :refer [ReadPort]]
       [cljs.pprint :as pp]
       [cljs.spec :as s])))

(defonce ^:private tables-ref (atom {}))
(defonce ^:private pid-count (atom 0))
(defonce ^:private table-generation (atom 1))
(defn- get-pid []
   (dec (swap! pid-count inc)))

(defn clear-tables! 
  "cleared table won't get updates from previously-started procs"
  [] 
  (reset! tables-ref {})
  (swap! table-generation inc)) 

(def ^:private print-keys
  [:pid :status :proc-name :start-time :end-time])

(defn- print-table*
  [table]
  (->> table
       (sort-by key)
       (map val)
       (pp/print-table print-keys)))

(defn print-table
  ([] (print-table ::global))
  ([table-name]
   (let [table-name (or table-name ::global)]
     (if-let [table (get @tables-ref table-name)]
       (print-table* table)
       (println (str "no table for " (pr-str table-name)))))))

(s/fdef cb-xf
        :args (s/cat :cb ifn?)
        :ref ifn?)

(defn- cb-xf
  [cb]
  (fn [xf]
    (fn
      ([] (xf))
      ([result] (cb) (xf result))
      ([result input] (xf result input)))))

(defn get-time
  []
  #?(:clj (.getTime (java.util.Date.))
     :cljs (.now js/Date)))  

(s/def ::namespaced-kw (s/and keyword? namespace))
(s/def ::table-name ::namespaced-kw)

(s/def ::spy-args
    (s/cat 
      :table-name (s/? ::table-name)  
      :opts (s/? (s/keys))
      :body (s/+ ::s/any)))

(s/fdef spy :args ::spy-args) 

(defn mark-start
  [{:keys [table-name pid start-time generation proc-name] :as opts}]
  (when (= generation @table-generation)
    (swap! tables-ref 
           update-in [table-name pid] 
           merge {:pid pid
                  :proc-name proc-name
                  :start-time start-time
                  :status ::started})))

(defn mark-closed
  [{:keys [table-name pid end-time generation :as opts]}]
  (when (= generation @table-generation)
    (swap! tables-ref
           update-in [table-name pid] 
           merge {:pid pid
                  :end-time end-time
                  :status ::closed})))

(defn mark-no-chan
  [{:keys [table-name pid end-time generation]}]
  (when (= generation @table-generation)
    (swap! tables-ref
           update-in [table-name pid] 
           merge {:pid pid
                  :end-time end-time
                  :status ::no-chan})))

(defn gen-proc-name
  [ns-str line column]
  (str "proc_" ns-str "_" line "_" column)) 

(defmacro spy
  [& forms*]
  (let [{:keys [table-name opts body]} (s/conform ::spy-args forms*)
        table-name (or (::table-name opts) table-name ::global)
        pid-cb (::pid-cb opts)
        ns-str (str *ns*)
        {:keys [line column]} (meta &form)
        proc-name (or (::proc-name opts) (gen-proc-name ns-str line column))]
    `(let [start-time# (get-time)
           pid# (get-pid)
           pid-cb# ~pid-cb
           data# {:proc-name ~proc-name
                  :pid pid#
                  :table-name ~table-name
                  :generation ~(deref table-generation)}] 
       (when pid-cb# (pid-cb# pid#))
       (mark-start (assoc data# 
                          :start-time start-time#))
       (let [out# (do ~@body)]
         (if (satisfies? ReadPort out#)
           (pipe out# (chan 1 (cb-xf #(mark-closed (assoc data# :end-time (get-time))))))
           (do (mark-no-chan (assoc data# :end-time (get-time))) 
               out#))))))
