(ns bloom.decant.core
  (:require
    [clojure.spec.alpha :as s]
    [clojure.string :as string]
    [clojure.core.match :as match]
    [spec-tools.data-spec :as ds]))

(defonce transactions (atom {}))

(s/def :decant/transactions
  (ds/spec
    {:name :decant/transactions
     :spec {keyword? {:params {keyword? (ds/or {:keyword keyword?
                                                :fn fn?
                                                :spec s/spec?})}
                      :rules fn? ;; must return array of true/false
                      :effect fn?}}}))

(defn register-transactions!
  [txs]
  {:pre [(s/valid? :decant/transactions txs)]}
  (reset! transactions
          (->> txs
               (map (fn [[k v]]
                      [k (assoc v :params-spec
                                (ds/spec {:name (keyword "tx-spec" (name k))
                                          :spec (v :params)}))]))
               (into {}))))

(defn- sanitize-params
  "Given a params-spec and params,
   if the params pass the spec, returns the params
     (eliding any extra keys)
   if params do not pass spec, returns nil"
  [tx params]
  (when (s/valid? (tx :params-spec) params)
    ;; TODO make use of spec shape to do a deep filter
    (select-keys params (keys (tx :params)))))

(defn- rule-errors
  "Returns boolean of whether the the rules for a transaction are satisfied.
   Should be called with sanitized-params."
  [tx sanitized-params]
  (->> ((tx :rules) sanitized-params)
       (remove (fn [[pass? _ _]] pass?))
       (map (fn [[pass? anomaly message]]
              {:anomaly anomaly
               :message message}))))

(defn explain-params-errors [spec value]
  (->> (s/explain-data spec value)
       ::s/problems
       (map (fn [{:keys [path pred val via in]}]
              (match/match [pred]
                           [([fn [_] ([contains? _ missing-key] :seq)] :seq)] {:issue :missing-key
                                                                               :key-path (conj path missing-key)}
                           [_] {:issue :incorrect-value
                                :key-path path})))
       (map (fn [{:keys [issue key-path]}]
              (str key-path " " issue)))
       (string/join "\n")))

(defn do! [tx-id params]
  (if-let [tx (@transactions tx-id)]
    (if-let [sanitized-params (sanitize-params tx params)]
      (let [errors (rule-errors tx sanitized-params)]
        (if (empty? errors)
          (do
            ((tx :effect) sanitized-params)
            true)
          (throw (ex-info (str "Transaction rules are not met:\n"
                               (string/join "\n" (map :message errors)))
                          {:anomaly :incorrect}))))
      (throw (ex-info (str "Transaction params do not meet spec:\n"
                           (explain-params-errors (tx :params-spec) params))
               {:anomaly :incorrect})))
    (throw (ex-info (str "No transaction with id " tx-id)
                    {:transaction-id tx-id
                     :anomaly :unsupported}))))
