(ns bloom.decant.core
  (:require
    [spec-tools.data-spec :as ds]
    [clojure.spec.alpha :as s]))

(defonce transactions (atom {}))

(s/def :decant/transactions
  (ds/spec
    {:name :decant/transactions
     :spec {keyword? {:params {keyword? (ds/or {:keyword keyword?
                                                :fn fn?
                                                :spec s/spec?})}
                      :rules fn? ;; must return array of true/false
                      :effect fn?}}}))

(defn register-transactions!
  [txs]
  {:pre [(s/valid? :decant/transactions txs)]}
  (reset! transactions
          (->> txs
               (map (fn [[k v]]
                      [k (assoc v :params-spec
                                (ds/spec {:name (keyword "tx-spec" (name k))
                                          :spec (v :params)}))]))
               (into {}))))

(defn- sanitize-params
  "Given a params-spec and params,
   if the params pass the spec, returns the params
     (eliding any extra keys)
   if params do not pass spec, returns nil"
  [tx params]
  (when (s/valid? (tx :params-spec) params)
    ;; TODO make use of spec shape to do a deep filter
    (select-keys params (keys (tx :params)))))

(defn- rules-pass?
  "Returns boolean of whether the the rules for a transaction are satisfied.
   Should be called with sanitized-params."
  [tx sanitized-params]
  (every? true? ((tx :rules) sanitized-params)))

(defn do! [tx-id params]
  (if-let [tx (@transactions tx-id)]
    (if-let [sanitized-params (sanitize-params tx params)]
      (if (rules-pass? tx sanitized-params)
        (do
          ((tx :effect) sanitized-params)
          true)
        (throw (ex-info "Transaction rules are not met" {:foo "bar"})))
      (throw (ex-info "Transaction params do not meet spec"
               {:explain (s/explain (tx :params-spec) params)})))
    (throw (ex-info (str "No transaction with id " tx-id) {:transaction-id tx-id}))))
