(ns bloom.commons.pages
  ";; call at some point to initialize:

   (initialize! [{:page/id :home
                  :page/view #'some-view-fn
                  :page/path \"/\"}
                 {:page/id :profile
                  :page/view (fn [[page-id {:keys [id]}]] ...)
                  :page/path \"/profile/:id\"
                  :page/parameters {:id integer?}}
                 ...])

   ;; include a reagent-view somewhere:
   [current-page-view]

   ;; to generate link string:
   (path-for [:profile {:id 5}])

   ;; to force navigation to some page
   (navigate-to! [:profile {:id 5}])

   ;; to check if some page is active
   (active? [:profile])
   (active? [:profile {:id 5}])
  "
  (:require
   [clojure.set :as set]
   [clojure.string :as string]
   [reitit.core :as reitit]
   [reitit.coercion :as coercion]
   [reitit.coercion.spec]
   [reitit.impl]
   #?@(:cljs
       [[accountant.core :as accountant]
        [reitit.frontend]
        [reagent.core :as r]]
       :clj
       [[ring.util.codec]]))
  (:import
   #?@(:cljs
       [(goog Uri)]
       :clj
       [(java.net URLEncoder URI)])))

(defonce router (atom nil))

#?(:cljs
   (defonce current-page (r/atom nil)))

(defn- url-encode [value]
  #?(:clj  (URLEncoder/encode ^String value "UTF-8")
     :cljs (js/encodeURIComponent value)))

(defn- query-string
  "Given map of params (allowing array values), returns a query string;
   Keys are sorted, and values within a key are sorted.
   (sorting is done to improve cacheability)"
  [params]
  (->> params
       sort
       (mapcat (fn [[k v]]
                 (cond
                   (or (vector? v) (set? v) (list? v))
                   (for [v' (sort v)]
                     (str (name k) "[]" "=" (url-encode v')))
                   :else
                   [(str (name k) "=" (url-encode v))])))
       (string/join "&")))

(defn classify-parameters
  "/a/:bar   {:bar __, :foo ___}
  =>
  {:path {:bar ___}, :query {:foo ___}}

  (assumes any params not in path are query params)"
  [path parameters]
  (let [all-params (set (keys parameters))
        path-params (:path-params (reitit.impl/parse path {}))
        query-params (set/difference all-params path-params)]
    {:path (select-keys parameters path-params)
     :query (select-keys parameters query-params)}))

(defn ->args [current-page]
  [(get-in current-page [:data :config :page/id])
   (merge (get-in current-page [:coerced-parameters :query])
          (get-in current-page [:coerced-parameters :path]))])

(defn make-router [pages]
  (reitit/router
   (->> pages
        (map (fn [page]
               [(page :page/path)
                {:name (page :page/id)
                 :coercion reitit.coercion.spec/coercion
                 :parameters (classify-parameters (page :page/path) (page :page/parameters))
                 :config page}])))
   {:compile coercion/compile-request-coercers}))

(defn query-params [uri]
  #?(:clj (or (some->> (.getQuery uri)
                       ring.util.codec/form-decode
                       (map (fn [[k v]]
                              [(keyword k) v]))
                       (into {}))
              {})
     :cljs (reitit.frontend/query-params uri)))

(defn match-by-path+ [router path]
  (let [uri #?(:clj (URI. path)
               :cljs (.parse Uri path))]
    (when-let [match (reitit/match-by-path router (.getPath uri))]
      (let [match (assoc match :query-params (query-params uri))]
        (assoc match :coerced-parameters (coercion/coerce! match))))))

(defn initialize!
  "Expects a list of pages, each a map with the following keys:
     :page/id           keyword, used in path-for

     :page/view         reagent view fn (recommend using #'view-fn)
                        receives params of page

     :page/path         string defining path
                        may include param patterns in path ex. /foo/:id
                        (params must also be included in :page/parameters)

     :page/parameters   map, data-spec coercion of path
                        any parameters not included in :page/path are assumed to be query-params

     :page/on-enter!    fn to call when page is navigated to
                        (right after reagent state is updated with new page-id)
                        receives params (of new page): [page-id params]

     :page/on-exit!     fn to call when page is navigated away from
                        receives params (of old page): [page-id params]"
  [pages]
  (when-not @router
    (reset! router (make-router pages))

    #?(:cljs
       (do
         (accountant/configure-navigation!
          {:nav-handler (fn [path]
                          (when-let [on-exit! (get-in @current-page [:data :config :page/on-exit!])]
                            (on-exit! (->args @current-page)))

                          (when-let [match (match-by-path+ @router path)]
                            (reset! current-page match)
                            (when-let [on-enter! (get-in @current-page [:data :config :page/on-enter!])]
                              (on-enter! (->args @current-page)))))
           :path-exists? (fn [path]
                           (let [uri (.parse Uri path)]
                             (boolean (reitit/match-by-path @router (.getPath uri)))))})
         (accountant/dispatch-current!)))))

(defn -path-for
  [router [page-id params]]
  (let [match (reitit/match-by-name router page-id params)
        query-params (select-keys params (keys (get-in match [:data :parameters :query])))]
    (str (:path match)
         (when (seq query-params)
           (let [query (query-string (params :query-params))]
             (when-not (string/blank? query)
               (str "?" query)))))))

(defn path-for
  [[page-id params]]
  (-path-for @router [page-id params]))

#?(:cljs
   (do
     (defn current-page-view []
       (when-let [view (get-in @current-page [:data :config :page/view])]
         [view (->args @current-page)]))

     (defn navigate-to!
       [[page-id params]]
       (accountant/navigate! (path-for [page-id params])))

     (defn active?
       [[page-id parameters]]
       (and
        (= page-id
           (get-in @current-page [:data :config :page/id]))
        (if parameters
          (= parameters
             (select-keys (merge (get-in @current-page [:coerced-parameters :query])
                                 (get-in @current-page [:coerced-parameters :path]))
                          (keys parameters)))
          true)))))
