(ns bloom.commons.ui.google-map
  (:require
    [clojure.set :refer [difference]]
    [clojure.string :as string]
    [goog.object :as o]
    [reagent.core :as r]))

(defn google-map-view
  "Accepts opts with the following structure:
   {:center {:lat -34.397
             :lng 150.644}
    :zoom 12
    :markers [{:id 1
               :lat -34
               :lng 150
               :view (fn [])}]}"
  [_]
  (let [marker-overlays (r/atom {})
        add-marker-overlays! (atom nil)
        remove-marker-overlays! (atom nil)
        rebound! (atom nil)]
    (r/create-class
      {:component-did-mount
       (fn [this]
         (let [opts (r/props this)
               google-map (js/google.maps.Map.
                            (r/dom-node this)
                            (clj->js {:center (opts :center)
                                      :zoom (opts :zoom)
                                      :clickableIcons false
                                      :mapTypeControl false
                                      :fullscreenControl false
                                      :streetViewControl false
                                      :zoomControlOptions {:position js/google.maps.ControlPosition.LEFT_BOTTOM
                                                           :style js/google.maps.ZoomControlStyle.SMALL}
                                      :styles [{:featureType "poi"
                                                :elementType "labels"
                                                :stylers [{:visibility "off"}]}]}))
               Overlay (fn [position]
                         (this-as this
                           (o/set this "position" position)
                           (o/set this "node" (.. js/document (createElement "div")))
                           (.. this -node -classList (add "marker"))
                           this))]
           (o/set Overlay "prototype" (js/google.maps.OverlayView.))
           (o/set (.-prototype Overlay) "onAdd"
                  (fn []
                    (this-as this
                      (.. this getPanes -floatPane (appendChild (.. this -node))))))
           (o/set (.-prototype Overlay) "onRemove" (fn []))
           (o/set (.-prototype Overlay) "draw"
                  (fn []
                    (this-as this
                      (let [xy (.. this getProjection (fromLatLngToDivPixel (o/get this "position")))]
                        (o/set (.. this -node -style) "position" "absolute")
                        (o/set (.. this -node -style) "left" (str (.-x xy) "px"))
                        (o/set (.. this -node -style) "top" (str (.-y xy) "px"))))))

           (reset! add-marker-overlays!
                   (fn [markers]
                     (swap! marker-overlays merge
                            (->> markers
                                 (map (fn [marker]
                                        [(marker :id)
                                         (let [latlng (js/google.maps.LatLng.
                                                        (marker :lat)
                                                        (marker :lng))
                                               marker-overlay (Overlay. latlng)]
                                           (.setMap marker-overlay google-map)
                                           marker-overlay)]))
                                 (into {})))))

           (reset! remove-marker-overlays!
                   (fn [markers]
                     (swap! marker-overlays
                            (fn [overlays]
                              (doseq [marker-overlay (vals (select-keys overlays (map :id markers)))]
                                (.setMap marker-overlay nil))
                              (apply dissoc overlays (map :id markers))))))

           (reset! rebound!
                   (fn [center markers]
                     (let [bounds (js/google.maps.LatLngBounds.)
                           ;; use a circle to set minimum bounds around the city center
                           circle (js/google.maps.Circle.
                                    #js {:center (clj->js center)
                                         ;; show on map for debugging:
                                         ;; :map google-map
                                         :radius 5000})]
                       (.union bounds (.getBounds circle))
                       (doseq [marker markers]
                         (.extend bounds #js {:lat (marker :lat)
                                              :lng (marker :lng)}))
                       (.fitBounds google-map bounds
                                   ;; padding to account for marker size
                                   #js {:left 15
                                        :top 15
                                        :right 150
                                        :bottom 15}))))

           (@add-marker-overlays! (opts :markers))
           (@rebound! (opts :center) (opts :markers))))

       :component-did-update
       (fn [this [_ prev-props]]
         (let [prev-opts prev-props
               new-opts (r/props this)]

           (let [prev-markers (:markers prev-opts)
                 new-markers (:markers new-opts)]
             (when (not= prev-markers new-markers)
               ;; must remove first, then add
               ;; because when 'editing' a marker, it shows up as stale & new, but with same id
               (let [stale-markers (difference (set prev-markers) (set new-markers))]
                 (@remove-marker-overlays! stale-markers))
               (let [new-markers (difference (set new-markers) (set prev-markers))]
                 (@add-marker-overlays! new-markers))))

           (@rebound! (new-opts :center) (new-opts :markers))))

       :reagent-render
       (fn [opts]
         [:<>
          [:<>
           (when (= (count (opts :markers))
                    (count @marker-overlays))
             (->> (opts :markers)
                  (map (fn [marker]
                         (when-let [node (o/get (@marker-overlays (marker :id)) "node")]
                           (js/ReactDOM.createPortal
                             (r/as-element (marker :view))
                             node))))
                  doall))]
          [:div.map]])})))
