(ns bloom.commons.textify
  (:require
    [clojure.string :as string]
    [clojure.walk :as walk]))

(defn strip-classes-and-ids [k]
  (keyword (first (string/split (name k) #"[.#]"))))

#_(strip-classes-and-ids :p.event) ; => :p

(defn normalize-hiccup-vector
  [v]
  (let [[attrs children] (if (map? (second v))
                           [(second v) (drop 2 v)]
                           [{} (rest v)])]
    {:element/tag (strip-classes-and-ids (first v))
     :element/attrs attrs
     :element/children children}))

(def tag-types
  {::block-element #{:html :body
                     :div
                     :address :article :aside :details :dialog :dd
                     :dt :figcaption :figure :footer
                     :header :hgroup :hr :li :main :nav :pre :section}
   ::inline-element #{:span} ;; handler treats unknown elements as inline
   ::block-element-with-padding #{:p :h1 :h2 :h3 :h4 :h6 :table
                                  :fieldset :form :ul :ol :dl :blockquote}})

(def tag->type
  (->> tag-types
       (mapcat (fn [[tag-type tags]]
                 (map (fn [tag]
                        [tag tag-type]) tags)))
       (into {})))

;; to make it easy to collapse multiple \n (ex. in case of nested divs)
;; use a special character, and then replace multiple of these with a single \n
(def collapsible-newline \u0091)

(defmulti element->text :element/tag)

(defmethod element->text :default
  [{:element/keys [tag children]}]
  (case (tag->type tag)
    ::block-element-with-padding
    (str collapsible-newline (string/join "" children) "\n")
    ::block-element
    (str collapsible-newline (string/join "" children))
    ::inline-element
    (string/join "" children)
    ;else
    (string/join "" children)))

(defmethod element->text :br
  []
  collapsible-newline)

(defmethod element->text :a
  [{:element/keys [attrs children]}]
  (str (string/join "" children) " (" (:href attrs) ")"))

(defn textify
  "Given hiccup, returns plain text version (with reasonable whitespace)"
  [content]
  (-> (walk/postwalk
        (fn [node]
          (cond
            (and
              (vector? node)
              (not (map-entry? node)))
            (element->text (normalize-hiccup-vector node))

            (seq? node)
            (clojure.string/join "" node)

            :else
            node))
        content)
      (string/replace (re-pattern (str collapsible-newline "{1,}")) "\n")
      (string/trim)))

#_(= (textify [:div
               [:div "A"]
               [:div "B"]])
     "A\nB")

#_(= (textify [:html
               [:body
                [:div "Hello"]
                [:div [:div "World" "XYZ"]]]])
     "Hello\nWorldXYZ")

#_(textify [:a {:href "asd"} "xyz"])

#_(= (textify [:html
               [:body
                [:h1 "Hello"]
                [:p "World" " " [:a {:href "link"} "XYZ"]]
                [:div "End"]]])
     "Hello\n\nWorld XYZ (link)\n\nEnd")
