(ns bloom.commons.html
  (:require
    [clojure.string :as string]
    [hiccup.core :as hiccup]))

(def exceptions
  ;; from https://github.com/preactjs/preact-compat/issues/222
  #{"accept-charset"
    "http-equiv"
    ;; svg
    "accent-height"
    "alignment-baseline"
    "arabic-form"
    "baseline-shift"
    "cap-height"
    "clip-path"
    "clip-rule"
    "color-interpolation"
    "color-interpolation-filters"
    "color-profile"
    "color-rendering"
    "fill-opacity"
    "fill-rule"
    "flood-color"
    "flood-opacity"
    "font-family"
    "font-size"
    "font-size-adjust"
    "font-stretch"
    "font-style"
    "font-variant"
    "font-weight"
    "glyph-name"
    "glyph-orientation-horizontal"
    "glyph-orientation-vertical"
    "horiz-adv-x"
    "horiz-origin-x"
    "marker-end"
    "marker-mid"
    "marker-start"
    "overline-position"
    "overline-thickness"
    "panose-1"
    "paint-order"
    "stop-color"
    "stop-opacity"
    "strikethrough-position"
    "strikethrough-thickness"
    "stroke-dasharray"
    "stroke-dashoffset"
    "stroke-linecap"
    "stroke-linejoin"
    "stroke-miterlimit"
    "stroke-opacity"
    "stroke-width"
    "text-anchor"
    "text-decoration"
    "text-rendering"
    "underline-position"
    "underline-thickness"
    "unicode-bidi"
    "unicode-range"
    "units-per-em"
    "v-alphabetic"
    "v-hanging"
    "v-ideographic"
    "v-mathematical"
    "vert-adv-y"
    "vert-origin-x"
    "vert-origin-y"
    "word-spacing"
    "writing-mode"
    "x-height"})

(defn kebab->camel [s]
  (if (contains? exceptions s)
    s
    (string/replace s #"(?<!data)(?<!aria)-(\w)" (fn [[_ match]] (string/capitalize match)))))

#_(kebab->camel "stop-color") ;; stop-color
#_(kebab->camel "data-foo") ;; data-foo
#_(kebab->camel "aria-foo") ;; aria-foo
#_(kebab->camel "everything-else") ;; everythingElse

(defn parse-opts
  "Given map of opts, rewrites any kebab-case opts to camelCase"
  [opts]
  (->> opts
       (map (fn [[k v]]
              [(kebab->camel (name k)) v]))
       (into {})))

(defn escape
  ;; based on https://github.com/weavejester/hiccup/blob/master/src/hiccup/util.clj#L80
  ;; but is it enough? https://wonko.com/post/html-escaping/
  [^String text]
  (.. text
      (replace "&" "&amp;")
      (replace "<" "&lt;")
      (replace ">" "&gt;")
      (replace "\"" "&quot;")))

(defn parse [component]
  (cond
    (string? component)
    (escape component)

    (not (sequential? component))
    component

    ; [some-fn args]
    (fn? (first component))
    (let [[f & args] component]
      (parse (apply f args)))

    ; [:div {} & body]
    (and (keyword? (first component))
         (map? (second component)))
    (let [[tag opts & body] component]
      (into [tag (parse-opts opts)]
            (map parse body)))

    ; [:div & body]
    (and (keyword? (first component)))
    (let [[tag & body] component]
      (parse (into [tag {}] body)))

    ; list of components
    :else
    (map parse component)))

(defn render [component]
  (hiccup/html
    (parse component)))
