(ns io.aviso.config
  "A system for reading and coercing configuration data.

  Configuration data is in the form of a *set* of files (mostly on the classpath) that follow a naming convention:

      conf/<profile>-<variant>.<extension>

  The list of profiles and variants is provided by the application.

  Currently, the extensions \"yaml\" and \"edn\" are supported.

  The configuration data is read from an appropriate set of such files, and merged together.
  The configuration is then passed through a Schema for validation and value coercion
  (for example, to convert strings into numeric types).

  Validation helps ensure that simple typos are caught early.
  Coercion helps ensure that the data is both valid and in a format ready to be consumed."
  (:require [schema.coerce :as coerce]
            [schema.utils :as su]
            [clj-yaml.core :as yaml]
            [clojure.edn :as edn]
            [clojure.string :as str]
            [io.aviso.tracker :as t]
            [io.aviso.toolchest.macros :refer [cond-let]]
            [clojure.java.io :as io]
            [medley.core :as medley]
            [com.stuartsierra.component :as component]
            [schema.core :as s])
  (:import (java.net URL)))

(defn- join-reader
  "An EDN reader used to join together a vector of values.

  Exposed as `#config/join`."
  [values]
  {:pre [(vector? values)]}
  (apply str values))

(defn- expand-env-var
  [source env-map expansion env-var default-value]
  (or (get env-map env-var)
      default-value
      (throw (ex-info (format "Unable to find expansion for `%s'." expansion)
                      {:env-var      env-var
                       :env-map-keys (keys env-map)
                       :source       source}))))

(defn- property-reader
  "An EDN reader used to convert either a string, or a vector of two strings, into a single
  string, using the properties assembled for the current invocation of
  [[assemble-configuration]].

  This is used as the EDN-compliant way to do substitutions, as per `${...}` syntax.

  A partial of this is exposed as `#config/prop`."
  [url env-map value]
  {:pre [(or (string? value)
             (and (vector? value)
                  (= 2 (count value))))]}
  (if (string? value)
    (expand-env-var url env-map value value nil)
    (let [[env-var default-value] value]
      (expand-env-var url env-map value env-var default-value))))


(defn- resources
  "For a given resource name on the classpath, provides URLs for all the resources that match, in
  no specific order."
  [name]
  (-> (Thread/currentThread) .getContextClassLoader (.getResources name) enumeration-seq))

(defn- split-env-ref
  [^String env-ref]
  (let [x (.indexOf env-ref ":")]
    (if (pos? x)
      [(subs env-ref 0 x)
       (subs env-ref (inc x))]
      [env-ref])))

(defn- expand-env-vars
  [env-map source]
  (str/replace source
               #"\$\{((?!\$\{).*?)\}"
               (fn [[expansion env-var-reference]]
                 (let [[env-var default-value] (split-env-ref env-var-reference)]
                   (expand-env-var source env-map expansion env-var default-value)))))

(defn- read-single
  "Reads a single configuration file from a URL, expanding environment variables, and
  then parsing the resulting string."
  [url parser env-map]
  (when url
    (t/track
      #(format "Reading configuration from `%s'." url)
      (->> (slurp url)
           (expand-env-vars env-map)
           (parser url env-map)))))

(defn- read-each
  "Read all resources matching a given path into a vector of parsed
  configuration data, ready to merge"
  [path parser env-map]
  (let [urls (resources path)]
    (keep #(read-single % parser env-map) urls)))

(defn- deep-merge
  "Merges maps, recursively. Collections accumulate, otherwise later values override
  earlier values."
  [existing new]
  (cond
    (map? existing) (merge-with deep-merge existing new)
    (coll? existing) (concat existing new)
    :else new))

(s/defschema ^{:added "0.1.10"}
  ConfigParser
  "A callback that is passed:

  * The URL of the resource.

  * The compiled environment and properties map

  * The text content of the resource (after processing expansions).

  The callback must return the parsed version of the content.

  Each extension is mapped to a corresponding ConfigParser.

  [[default-extensions]] provides the default mappings."
  (s/=> s/Any URL {s/Str s/Str} s/Str))

(def default-extensions
  "The default mapping from file extension to a [[ConfigParser]] for content from such a file.

  Provides parsers for the \"yaml\" and \"edn\" extensions."
  {"yaml" (fn [_ _ content]
            (yaml/parse-string content true))
   "edn"  (fn [uri env-map content]
            (edn/read-string {:readers {'config/join join-reader
                                        'config/prop (partial property-reader uri env-map)}}
                             content))})

(s/defschema ^{:added "0.1.10"} ResourcePathSelector
  "Map of values passed to a [[ResourcePathGenerator]]."
  {:profile   s/Keyword
   :variable  (s/maybe s/Keyword)
   :extension s/Str})

(s/defschema ^{:added "0.1.10"} ResourcePathGenerator
  "A callback that converts a configuration file [[ResourcePathSelector]]
  into some number of resource path strings.

  The standard implementation is [[default-resource-path]]."
  (s/=> [String] ResourcePathSelector))

(s/defn default-resource-path :- [String]
  "Default mapping of a resource path from profile, variant, and extension.
  A single map is passed, with the following keys:

  :profile - keyword
  : profile to add to path

  :variant - keyword
  : variant to add to the path, or nil

  :extension - string
  : extension (e.g., \"yaml\")

  The result is typically \"conf/profile-variant.ext\".

  However, \"-variant\" is omitted when variant is nil.

  Since 0.1.10, returns a seq of files.
  Although this implementation only returns a single value, supporting a seq of values
  makes it easier to extend (rather than replace) the default behavior with an override."
  [selector :- ResourcePathSelector]
  (let [{:keys [profile variant extension]} selector]
    [(str "conf/"
          (->> [profile variant]
               (remove nil?)
               (mapv name)
               (str/join "-"))
          "."
          extension)]))

(def ^{:added "0.1.9"} default-variants
  "The default list of variants. The combination of profile and variant is the main way
  that resource file names are created (combined with a fixed prefix and a supported
  extension).

  The order of the variants determine load order, which is relevant.

  A nil variant is always prefixed to this list; this represents loading default
  configuration for the profile.

  Typically, a library creates a component or other entity that is represented within
  config as a profile.

  The local variant may be used for test-specific overrides, or overrides for a user's
  development (say, to redirect a database connection to a local database), or even
  used in production."
  [:local])

(defn- get-parser [^String path extensions]
  (let [dotx      (.lastIndexOf path ".")
        extension (subs path (inc dotx))]
    (or (get extensions extension)
        (throw (ex-info "Unknown extension for configuration file."
                        {:path       path
                         :extensions extensions})))))

(defn merge-value
  "Merges a command-line argument into a map. The command line argument is of the form:

       path=value

   where path is the path to value; it is split at slashes and the
   individual strings converted to keywords.

   e.g.

       web-server/port=8080

   is equivalent to

       (assoc-in m [:web-server :port] \"8080\")

   "
  {:since "0.1.1"}
  [m arg]
  (cond-let
    [[_ path value] (re-matches #"([^=]+)=(.*)" arg)]

    (not path)
    (throw (IllegalArgumentException. (format "Unable to parse argument `%s'." arg)))

    [keys (map keyword (str/split path #"/"))]

    :else
    (assoc-in m keys value)))

(defn- parse-args
  [args]
  (loop [remaining-args   args
         additional-files []
         overrides        {}]
    (cond-let
      (empty? remaining-args)
      [additional-files overrides]

      [arg (first remaining-args)]

      (= "--load" arg)
      (let [[_ file-name & more-args] remaining-args]
        (recur more-args (conj additional-files file-name) overrides))

      :else
      (recur (rest remaining-args)
             additional-files
             (merge-value overrides arg)))))

(s/defschema ^{:added "0.1.10"} AssembleOptions
  "Defines the options passed to [[assemble-configuration]]."
  {(s/optional-key :schemas)          [s/Any]
   (s/optional-key :additional-files) [s/Str]
   (s/optional-key :args)             [s/Str]
   (s/optional-key :overrides)        s/Any
   (s/optional-key :profiles)         [s/Keyword]
   (s/optional-key :properties)       {s/Any s/Str}
   (s/optional-key :variants)         [(s/maybe s/Keyword)]
   (s/optional-key :resource-path)    ResourcePathGenerator
   (s/optional-key :extensions)       {s/Str ConfigParser}})

(s/defn assemble-configuration
  "Reads the configuration, as specified by the options.

  Inside each configuration file, the content is scanned for property expansions.

  Expansions allow environment variables, JVM system properties, or explicitly specific properties
  to be substituted into the content of a configuration file, *before* it is parsed.

  Expansions take two forms:

  * `${ENV_VAR}`
  * `${ENV_VAR:default-value}`

  In the former case, a non-nil value for the indicated property or environment variable
  must be available, or an exception is thrown.

  In the later case, a nil value will be replaced with the indicated default value.  e.g. `${HOST:localhost}`.

  The :args option is passed command line arguments (as from a -main function). The arguments
  are used to add further additional files to load, and provide additional overrides.

  Arguments are either \"--load\" followed by a path name, or \"path=value\".

  In the second case, the path and value are as defined by [[merge-value]].

  :schemas
  : A seq of schemas; these will be merged to form the full configuration schema.

  :additional-files
  : A seq of absolute file paths that, if they exist, will be loaded last, after all
    normal resources.
    This is typically used to provide an editable (outside the classpath) file for final
    production configuration overrides.

  :args
  : Command line arguments to parse; these yield yet more additional files and
    the last set of overrides.

  :overrides
  : A map of configuration data that is overlayed (using a deep merge)
    on the configuration data read from the files, before validation and coercion.

  :profiles
  : A seq of keywords that identify which profiles should be loaded and in what order.
    The default is an empty list.

  :properties
  : An optional map of properties that may be substituted, just as environment
    variable or System properties can be. Explicit properties have higher precendence than JVM
    system properties, which have higher precendence than environment
    variables; however the convention is that environment variable names
    are all upper case, and properties are all lower case, so actual conflicts
    should not occur.
  : The keys of the properties map are converted to strings via `name`, so they
    may be strings or symbols, or more frequently, keywords.
  : Most often the properties map is used for specific overrides in testing, or
    to expose some bit of configuration that cannot be directly extracted
    from environment variables or JVM system properties.

  :variants
  : The variants searched for, for each profile.
  : [[default-variants]] provides the default list of variants.  A nil variant
    is always prefixed on the provided list.

  :resource-path
  : A function that builds resource paths from  profile, variant, and extension.
  : The default is [[default-resource-path]], but this could be overridden
    to (for example), use a directory structure to organize configuration files
    rather than splitting up the different components of the name using dashes.

  :extensions
  : Maps from extension (e.g., \"yaml\") to an appropriate parsing function.

  Any additional files are loaded after all profile and variant files.

  Files specified via `--load` arguments are then loaded.

  The contents of each file are deep-merged together; later files override earlier files.

  Overrides via the :overrides key are applied, then overrides from command line arguments
  (provided in the :args option) are applied."
  [options :- AssembleOptions]
  (let [{:keys [schemas overrides profiles variants
                resource-path extensions additional-files
                args properties]
         :or   {extensions    default-extensions
                variants      default-variants
                profiles      []
                resource-path default-resource-path}} options
        env-map       (-> (sorted-map)
                          (into (System/getenv))
                          (into (System/getProperties))
                          (into (medley/map-keys name properties)))
        [arg-files arg-overrides] (parse-args args)
        variants'     (cons nil variants)
        raw           (for [profile profiles
                            variant variants'
                            [extension parser] extensions
                            path    (resource-path {:profile   profile
                                                    :variant   variant
                                                    :extension extension})]
                        (read-each path parser env-map))
        flattened     (apply concat raw)
        extras        (for [path (concat additional-files arg-files)
                            :let [parser (get-parser path extensions)]]
                        (read-single (io/file path) parser env-map))
        conj'         (fn [x coll] (conj coll x))
        merged        (->> (concat flattened extras)
                           vec
                           (conj' overrides)
                           (conj' arg-overrides)
                           (apply merge-with deep-merge))
        merged-schema (apply merge-with deep-merge schemas)
        coercer       (coerce/coercer merged-schema coerce/string-coercion-matcher)
        config        (coercer merged)]
    (if (su/error? config)
      (throw (ex-info (str "The configuration is not valid: " (-> config su/error-val pr-str))
                      {:schema  merged-schema
                       :config  merged
                       :failure (su/error-val config)}))
      config)))

(defprotocol Configurable
  (configure [this configuration]
    "Passes a component's individual configuration to the component,
    as defined by the three-argument version of [[with-config-schema]].

    When this is invoked (see [[configure-components]]),
    a component's dependencies *will* be available, but in an un-started
    state."))

(defn with-config-schema
  "Adds metadata to the component to define the configuration schema for the component.

  The two arguments version is the original version.
  When using this approach, in concert with [[extend-system-map]], the
  component should have a dependency on the configuration component
  (typically added to the system map as key :configuration).
  The configuration componment will be the configuration map, created via [[assemble-configuration]].

  The modern alternative is the three argument variant.
  This defines a top-level configuration key (e.g., :web-service)
  and a schema for just that key.

  The component will receive *just* that configuration in its
  :configuration key.

  Alternately, the component may implement the
  [[Configurable]] protocol. It will be passed just its own configuration."
  ([component schema]
   (vary-meta component assoc ::schema schema))
  ([component config-key schema]
   (vary-meta component assoc ::config-key config-key
              ;; This is what's merged to form the master schema
              ::schema {config-key schema})))

(defn extract-schemas
  "For a seq of components (the values of a system map),
   extracts the schemas associated via [[with-config-schema]], returning a seq of schemas."
  [components]
  (keep (comp ::schema meta) components))

(defn extend-system-map
  "Uses the system map and options to read the configuration, using [[assemble-configuration]].
  Returns the system map with one extra component, the configuration
  itself (ready to be injected into the other components)."
  {:added "0.1.9"}
  ([system-map options]
   (extend-system-map system-map :configuration options))
  ([system-map configuration-key options]
   (let [schemas       (-> system-map vals extract-schemas)
         configuration (t/track "Reading configuration."
                                (assemble-configuration (assoc options :schemas schemas)))]
     (assoc system-map configuration-key configuration))))

(defn- apply-configuration
  [component full-configuration]
  (cond-let
    [config-key (-> component meta ::config-key)]

    (nil? config-key)
    component

    [component-configuration (get full-configuration config-key)]

    (satisfies? Configurable component)
    (configure component component-configuration)

    :else
    (assoc component :configuration component-configuration)))

(defn configure-components
  "Configures the components in the system map, returning an updated system map.

  In the simple case, the configuration is expected to be in the :configuration
  key of the system map (the default when using [[extend-system-map]].

  In the two argument case, the system configuration map is supplied as the second parameter.

  Typically, this should be invoked *before* the system is started, as most
  components are expected to need configuration in order to start."
  {:added "0.1.9"}
  ([system-map]
   (configure-components system-map (:configuration system-map)))
  ([system-map configuration]
   (component/update-system system-map
                            (keys system-map)
                            apply-configuration
                            configuration)))
