(ns ion-torrent-api.core
  (:require [clj-http.client :as client]
            [clojure.java.io :as io]))

;;; experiment:-
;;; - set of samples
;;; - run name
;;; - image/block processing data
;;; - list of results, one for each plugin that was run
;;; eg, exp 50 has results:
;;; 77 pluginStore: IonReporterUploader ; pluginResults: /rundb/api/v1/pluginresult/89/
;;; 61 pluginStore: Alignment  ; pluginResults: /rundb/api/v1/pluginresult/60/
;;; 62 pluginStore: IonReporterUploader / coverageAnalysis / variantCaller
;;;    pluginResults:     "/rundb/api/v1/pluginresult/71/", "/rundb/api/v1/pluginresult/66/", "/rundb/api/v1/pluginresult/61/"
;;; /rundb/api/v1/pluginresult/71/ : "plugin" : {"name": "IonReporterUploader"}
;;; /rundb/api/v1/pluginresult/66/ : "plugin" : {"name": "coverageAnalysis"}
;;;  "path": "/results/analysis/output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/plugin_out/coverageAnalysis_out",
;;;  "reportLink": "/output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/",
;;; 
;;; path to amplicon coverage file is "reportLink" + tail part of "path":-
;;;  /output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/plugin_out/coverageAnalysis_out/IonXpressRNA_001/IonXpressRNA_001_R_2013_06_03_23_30_18_user_AIB-24-AmpliSeq_CCP_24_Auto_user_AIB-24-AmpliSeq_CCP_24_50.amplicon.cov.xls
;;;
;;; 

;;; /rundb/api/v1/pluginresult/61/ : "plugin" : {"name": ""}

;;; pluginresult for Variant Caller:-

;; "store": {
;;     "Aligned Reads": "R_2013_06_03_23_30_18_user_AIB-24-AmpliSeq_CCP_24",
;;     "Configuration": "Somatic - Low Stringency",
;;     "Library Type": "AmpliSeq",
;;     "Target Loci": "Not using",
;;     "Target Regions": "4477685_Comprehensive_CCP_bedfile_20120517",
;;     "Trim Reads": true,
;;     "barcoded": "true",
;;     "barcodes": {
;;         "IonXpressRNA_001": {
;;             "hotspots": { },
;;             "variants": {
;;                 "het_indels": 88,
;;                 "het_snps": 1733,
;;                 "homo_indels": 34,
;;                 "homo_snps": 282,
;;                 "no_call": 0,
;;                 "other": 1,
;;                 "variants": 2138
;;             }
;;         },

;;; pluginresult for Coverage Analysis:-

;; store": {
;;     "Non-duplicate": "No",
;;     "Target padding": "0",
;;     "Targetted regions": "/results/uploads/BED/1/hg19/merged/plain/4477685_Comprehensive_CCP_bedfile_20120517.bed",
;;     "Uniquely mapped": "No",
;;     "barcoded": "true",
;;     "barcodes": {
;;         "IonXpressRNA_001": {
;;             "Alignments": "IonXpressRNA_001_R_2013_06_03_23_30_18_user_AIB-24-AmpliSeq_CCP_24_Auto_user_AIB-24-AmpliSeq_CCP_24_50",
;;             "Amplicons reading end-to-end": "8.67%",
;;             "Amplicons with at least 1 read": "99.51%",
;;             "Amplicons with at least 100 reads": "90.96%",
;;             "Amplicons with at least 20 reads": "96.94%",
;;             "Amplicons with at least 500 reads": "53.41%",
;;             "Amplicons with no strand bias": "92.97%",
;;             "Average base coverage depth": "671.9",
;;             "Average reads per amplicon": "794.0",
;;             "Bases in target regions": "1688650",
;;             "Number of amplicons": "15992",
;;             "Number of mapped reads": "13541550",
;;             "Percent assigned amplicon reads": "93.77%",
;;             "Percent base reads on target": "95.73%",
;;             "Percent reads on target": "93.77%",
;;             "Reference (File)": "hg19",
;;             "Target base coverage at 100x": "88.24%",
;;             "Target base coverage at 1x": "99.34%",
;;             "Target base coverage at 20x": "96.51%",
;;             "Target base coverage at 500x": "47.27%",
;;             "Target bases with no strand bias": "79.94%",
;;             "Targeted Regions": "4477685_Comprehensive_CCP_bedfile_20120517",
;;             "Total aligned base reads": "1185203398",
;;             "Total assigned amplicon reads": "12697352",
;;             "Total base reads on target": "1134571007",
;;             "Uniformity of amplicon coverage": "84.72%",
;;             "Uniformity of base coverage": "83.69%",
;;             "Using": "All Mapped Reads"
;;         },

;;; plugin:-
;;; "id": 36, "name": "coverageAnalysis", "resource_uri": "/rundb/api/v1/plugin/36/",
;;; "version": "3.6.63324", "versionedName": "coverageAnalysis--v3.6.63324"
;;;
;;; "id": 37, "name": "variantCaller", "resource_uri": "/rundb/api/v1/plugin/37/",
;;; "version": "3.6.63335", "versionedName": "variantCaller--v3.6.63335"

;;; experiment fields:-

;;; (:status :notes :sample :date :storageHost :chipBarcode :plan
;;; :user_ack :eas_set :log :isReverseRun :runtype :samples
;;; :seqKitBarcode :expDir :storage_options :flowsInOrder :resultDate
;;; :sequencekitbarcode :pgmName :sequencekitname :displayName
;;; :reagentBarcode :unique :baselineRun :resource_uri :autoAnalyze
;;; :reverse_primer :flows :star :rawdatastyle :results :cycles
;;; :ftpStatus :runMode :expName :metaData :id :usePreBeadfind
;;; :chipType :expCompInfo :diskusage)

;;; results fields:-
;;; 
;;; (:status :planShortID :tfFastq :bamLink :fastqLink
;;; :analysisVersion :tfmetrics :reportStatus :reportstorage :projects
;;; :log :sffLink :pluginresults :runid :analysismetrics
;;; :qualitymetrics :timeToComplete :eas :autoExempt :pluginStore
;;; :processedCycles :resource_uri :tfSffLink :resultsType
;;; :representative :filesystempath :libmetrics :pluginState
;;; :reference :timeStamp :resultsName :processedflows :parentIDs
;;; :reportLink :framesProcessed :metaData :id :experiment :diskusage)
;;;
;;; options:-
;;; limit offset
;;; sample status codes : "Completed" "Completed with 1 error(s)"

(defn- pluginresult-api-path
  "API path to pluginresult files."
  [res]
  (let [{path "path"
         report-link "reportLink"} res]
    ;; sample path:
    ;;   "/results/analysis/output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/plugin_out/coverageAnalysis_out"
    ;; sample reportLink:
    ;;   "/output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/"
    ;; required API path to report files:
    ;;   "/output/Home/Auto_user_AIB-24-AmpliSeq_CCP_24_50_061/plugin_out/coverageAnalysis_out"
    (.substring path (.indexOf path report-link))))

(defn- ensure-starts-with
  "Ensure s starts with prefix."
  [prefix s]
  (if (.startsWith s prefix) s (str prefix s)))

(defn resource-file
  "Return a file from host."
  [host creds file-path]
  (:body (client/get (str "http://" host file-path)
                     {:basic-auth creds})))

(defn write-resource-file
  "write a file from host."
  [host creds file-path dest-file]
  (with-open [o (io/output-stream dest-file)]
    (io/copy (:body (client/get (str "http://" host file-path)
                                {:as :stream :basic-auth creds}))
             o
             :buffer-size (* 16 1024))
    dest-file))

(defn resource
  "Return a JSON resource from host.
Keys are not coerced to keywords as the JSON keys can have spaces in them which are invalid as keywords and not printable+readable."
  [host creds resource & [opts]]
  (:body (client/get (str "http://" host (ensure-starts-with "/rundb/api/v1/" resource))
                     {:as :json-string-keys :basic-auth creds :query-params opts})))

(defn experiment
  "Experiments that have run."
  [host creds & [opts]]
  (resource "experiment/" host creds (assoc opts "status__exact" "run")))

(defn experiment-name
  "Experiment by name."
  [host creds name & [opts]]
  (let [{{tot "total_count"} "meta"
         exp "objects"} (resource host creds "experiment/" (merge opts {"expName__exact" name "status__exact" "run"}))]
    (if (= 1 tot) (first exp))))

(defn results
  "Results that have completed."
  [host creds & [opts]]
  (resource host creds "results/" (assoc opts "status__startswith" "Completed")))

(defn experiment-results
  "Results that have completed for an experiment and are not thumbnails."
  [host creds exp]
  (remove #(get-in % ["metaData" "thumb"])
          (map #(resource host creds % {"status__startswith" "Completed"}) (get exp "results"))))

(defn experiment-pluginresults
  "Plugin results that have completed for an experiment."
  [host creds exp]
  (map #(resource host creds % {"status__exact" "Completed"})
       (mapcat #(get % "pluginresults") (experiment-results host creds exp))))

(defn experiment-coverage
  "coverageAnalysis plugin results that have completed, for an experiment."
  [host creds exp]
  (filter #(-> % (get-in ["plugin" "name"]) (= "coverageAnalysis"))
          (experiment-pluginresults host creds exp)))

(defn experiment-variants
  "variantCaller plugin results that have completed, for an experiment."
  [host creds exp]
  (filter #(-> % (get-in ["plugin" "name"]) (= "variantCaller"))
          (experiment-pluginresults host creds exp)))

(defn pluginresult
  "Pluginresult that have completed."
  [host creds & [opts]]
  (resource host creds "pluginresult/"  (assoc opts "status__startswith" "Completed")))

(defn pluginresult-id
  "Pluginresult that have completed."
  [host creds id]
  (resource host creds (str "pluginresult/" id "/")))

(defn coverage
  "coverageAnalysis for id."
  [host creds id]
  (let [{{name "name"} "plugin" :as res} (pluginresult-id host creds id)]
    (if (= "coverageAnalysis" name) res)))

(defn variant-call
  "variantCall for id."
  [host creds id]
  (let [{{name "name"} "plugin" :as res} (pluginresult-id host creds id)]
    (if (= "variantCaller" name) res)))


;;; (def p66-001 (ion/amplicon-coverage-file (ion/pluginresult-id host2 user-pass 66) "IonXpressRNA_001"))
;;; (count (ion/resource-file p66-001 host2 user-pass))

(defn coverage-amplicon-file-path
  "Coverage by amplicon file path. Barcode is a keyword or string."
  [cov barcode]
  (let [{{{{prefix "Alignments"} (name barcode)} "barcodes"} "store"} cov
        path (str (pluginresult-api-path cov) "/" (name barcode) "/" prefix ".amplicon.cov.xls")]
    path))

(defn tsvc-variant-file-path
  "TSVC variant vcf.gz file path. Barcode is a keyword or string."
  [res barcode]
  (let [path (str (pluginresult-api-path res) "/" (name barcode) "/TSVC_variants.vcf.gz" )]
    path))

(defn tsvc-variant-tbi-file-path
  "TSVC variant vcf.gz.tbi file path. Barcode is a keyword or string."
  [res barcode]
  (let [path (str (pluginresult-api-path res) "/" (name barcode) "/TSVC_variants.vcf.gz.tbi" )]
    path))



#_(defn coverage-amplicon-files
  [cov]
  (map #(coverage-amplicon-file-path cov (name %)) (-> cov (get-in ["store" "barcodes"]) keys)))

(comment
  (def host "aibn-iontorrent-2.aibn.uq.edu.au")
  (def creds ["ionadmin" "ionadmin"])
  (def e6n "R_2013_03_11_23_41_27_user_AIB-6-Ion_AmpliSeq_Comprehensive_Cancer_Panel")
  (def e6 (ion/experiment-name host creds e6n))
  (def e6v (ion/experiment-variants host creds e6))
  (def e6c (ion/experiment-coverage host creds e6))
  )
