;   This file is part of clj-docker-client.
;
;   clj-docker-client is free software: you can redistribute it and/or modify
;   it under the terms of the GNU Lesser General Public License as published by
;   the Free Software Foundation, either version 3 of the License, or
;   (at your option) any later version.
;
;   clj-docker-client is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
;   GNU Lesser General Public License for more details.
;
;   You should have received a copy of the GNU Lesser General Public License
;   along with clj-docker-client. If not, see <http://www.gnu.org/licenses/>.

(ns clj-docker-client.requests
  (:require [clojure.string :as s]
            [jsonista.core :as json]
            [unixsocket-http.core :as uhttp])
  (:import (java.io InputStream)
           (java.time Duration)
           (java.net URI)
           (java.util.regex Pattern)))

(defn panic!
  "Helper for erroring out for wrong args."
  [^String message]
  (throw (IllegalArgumentException. message)))

(defn- unix-socket-client
  [path timeouts]
  (uhttp/client
    path
    {:read-timeout-ms (:read-timeout timeouts 0)
     :write-timeout-ms (:write-timeout timeouts 0)
     :connect-timeout-ms (:connect-timeout timeouts 0)
     :call-timeout-ms (:call-timeout timeouts 0)}))

(defn connect*
  "Connects to the provided :uri in the connection options.

  Optionally takes connect, read, write and call timeout in ms.
  All are set to 0 by default, which is no timeout.

  Returns the connection.

  The url must be fully qualified with the protocol.
  eg. unix:///var/run/docker.sock or https://my.docker.host:6375"
  [{:keys [uri timeouts]}]
  (when (nil? uri)
    (panic! ":uri is required"))
  (let [uri    (URI. uri)
        scheme (.getScheme uri)
        path   (.getPath uri)
        client (case scheme
                 "unix" (unix-socket-client path timeouts)
                 (panic! (format "Protocol '%s' not supported yet." scheme)))]
    {:client client}))

(defn interpolate-path
  "Replaces all occurrences of {k1}, {k2} ... with the value map provided.

  Example:
  given a/path/{id}/on/{not-this}/root/{id} and {:id hello}
  results in: a/path/hello/{not-this}/root/hello."
  [path value-map]
  (let [[param value] (first value-map)]
    (if (nil? param)
      path
      (recur
       (s/replace path
                  (re-pattern (format "\\{([%s].*?)\\}"
                                      (-> param name Pattern/quote)))
                  value)
       (dissoc value-map param)))))

(defn build-request
  "Builds a request that can be used with our unixsocket-http client."
  [{:keys [conn method ^String url query header path body as]}]
  {:pre [(contains? #{:get :head :put :post :delete :patch nil} method)]}
  (-> {:client       (:client conn)
       :method       (or method :get)
       :url          (interpolate-path url path)
       :query-params query
       :headers      (into {} header)
       :body         body
       :as           (or as :string)}
      (cond->
        (map? body)
        (-> (update :body json/write-value-as-string)
            (assoc-in [:headers "content-type"] "application/json"))
        (instance? InputStream body)
        (assoc-in [:headers "content-type"] "application/octet-stream"))))

(defn fetch
  "Performs the request.

  If :as is passed as :stream, returns an InputStream.
  If passed as :socket, returns the underlying UNIX socket for direct I/O."
  [{:keys [conn as throw-exception?] :as args}]
  (-> (build-request args)
      (uhttp/request)
      (:body)))

(comment
  (unix-socket-client-builder "/var/run/docker.sock")

  (stream->req-body (java.io.FileInputStream. (java.io.File. "README.md")))

  (Pattern/quote "\\{([a-id].*?)\\}")

  (interpolate-path "a/{id}/path/to/{something-else}/and/{xid}/{not-this}"
                    {:id             "a-id"
                     :xid            "b-id"
                     :something-else "stuff"})

  (fetch {:conn (connect* {:uri "unix:///var/run/docker.sock"})
          :url  "/_ping"}))
