(ns clj-progress.example)

(use 'clj-progress.core)

(defn process-item [item]
  (-> (rand)
      (* 1000)
      int
      Thread/sleep)
  item)

(defn greedy []
  (->>  (range 50)
        (init "Processing")
        (map (comp tick process-item))
        (reduce +)
        done))

(defn lazy []
  (->>  (iterate inc 0)
        (init "Processing" 50)
        (map (comp tick process-item))
        (take 50)
        (reduce +)
        done))


(set-progress-bar! "  downloading [:bar] :percent :etas")

(config-progress-bar!
  :complete   \#
  :current    \#
  :incomplete \=)

(use 'clj-progress.core)

(defn progress []
  (init 50)
  (reduce + (map  (comp tick process-item)
                  (range 50)))
  (done))

(defn update-atom [state data]
  (swap! state merge data))

(defn atom-handler [a]
  { :init (partial update-atom a)
    :tick (partial update-atom a)
    :done (fn [_] (swap! a assoc :ready true))})

(defn atomic []
  (let [state (atom {:ready false})]
    (with-progress-handler (atom-handler state)
      (future (progress)))
    state))

(defn multi []
  (let [atoms (repeatedly 5 atomic)]
    (while (some (comp not :ready deref) atoms)
      (dotimes [i 5]
        (let [{:keys [ttl done]} (deref (nth atoms i))]
          (println i ":" done "/" ttl)))
      (println "==========")
      (Thread/sleep 1000))
    (println "All done!")))

(use 'clj-progress.core)
(use 'clojure.java.io)

(defn copy-file [input-path output-path]
  (init (-> (file input-path) .length))
  (with-open [input  (input-stream  input-path )
              output (output-stream output-path)]
    (let [buffer (make-array Byte/TYPE 1024)]
      (loop []
        (let [size (.read input buffer)]
          (when (pos? size)
            (.write output buffer 0 size)
            (tick-by size)
            (recur))))))
  (done))

(defn uncertain []
  (with-progress-bar "  downloading [:wheel][:bar] :percent :etas"
    (init -1)
    (reduce + (map  (comp tick process-item)
                    (range 50)))
    (done)))

