;;; Copyright ©️ Rachel Bowyer 2022, 2023. All rights reserved.
;;;
;;; This program and the accompanying materials
;;; are made available under the terms of the Eclipse Public License v2.0
;;; which accompanies this distribution, and is available at
;;; https://www.eclipse.org/legal/epl-2.0/
;;;
;;; This Source Code may also be made available under the following
;;; Secondary Licenses when the conditions for such availability set forth
;;; in the Eclipse Public License, v. 2.0 are satisfied: GNU General Public
;;; License as published by the Free Software Foundation, either version 2
;;; of the License, or (at your option) any later version, with the GNU
;;; Classpath Exception which is available at
;;; https://www.gnu.org/software/classpath/license.html.


(ns info.bowyer.sam-diff.optimised.meyer-string
  (:require
    [info.bowyer.sam-diff.formatter.string :as formatter]
    [info.bowyer.sam-diff.optimised.post-process :refer [post-process]])
  (:import [info.bowyer.sam_diff_java MeyerString MeyerString$MeyerStringDiffInfo
            MeyerString$EditType MeyerString$EditInstruction]
           [info.bowyer.sam_diff_java Type]))

(defn edit-distance
  [a b]
  (MeyerString/editDistance a b))

(defn bidirectional-edit-distance
  [a b]
  (MeyerString/bidirectionalEditDistance a b))

(defmethod post-process Type/MeyerString
  [^MeyerString$MeyerStringDiffInfo diff-info]
  (let [a (.getA diff-info)
        b (.getB diff-info)]
    (loop [^MeyerString$EditInstruction [f & r] (.getEditInstructions diff-info)
           result {:inserted    {}
                   :deleted     {}
                   :levenshtein (.getDistance diff-info)
                   :type        :string
                   :a           a}]
      (if (nil? f)
        result
        (recur r
               (let [x (.getX f)
                     y (.getY f)
                     t (.getType f)]
                 (cond
                   (= t MeyerString$EditType/DELETE)
                   (assoc-in result [:deleted (- (count a) x)] (.charAt a (dec x)))

                   (= t MeyerString$EditType/INSERT)
                   (update-in result [:inserted (- (count a) x)]
                              #(conj (or % []) (.charAt b (dec y)))))))))))

(defn raw-diff
  "Convenience method"
  [a b]
  (some-> (MeyerString/diff a b) post-process))


(defn diff
  "Convenience method"
  [a b]
  (let [diff-info (MeyerString/diff a b)]
    (some-> diff-info
            post-process
            (formatter/frmt {}))))




