;;; Copyright ©️ Rachel Bowyer 2022, 2023. All rights reserved.
;;;
;;; This program and the accompanying materials
;;; are made available under the terms of the Eclipse Public License v2.0
;;; which accompanies this distribution, and is available at
;;; https://www.eclipse.org/legal/epl-2.0/
;;;
;;; This Source Code may also be made available under the following
;;; Secondary Licenses when the conditions for such availability set forth
;;; in the Eclipse Public License, v. 2.0 are satisfied: GNU General Public
;;; License as published by the Free Software Foundation, either version 2
;;; of the License, or (at your option) any later version, with the GNU
;;; Classpath Exception which is available at
;;; https://www.gnu.org/software/classpath/license.html.


(ns info.bowyer.sam-diff.optimised.dijkstra-string
  "Wrapper functions around a Dijksta based string comparison.
   It is a simple algorithm that is relatively easy to see is correct.
   It forms the basis of the more complicated variable weight bi-directional
   Dijksta non-atomic sequence algorithm.
   It is also used as a comparison in property based testing the more complicated
   algorithms."
  (:require
    [info.bowyer.sam-diff.formatter.string :as format]
    [info.bowyer.sam-diff.optimised.post-process :refer [post-process]])
  (:import [info.bowyer.sam_diff_java  DijkstraString DijkstraString$DijkstraStringDiffInfo DijkstraString$QueueElement
                                      DijkstraString$Direction Type]))

(defmethod post-process Type/DijkstraString
  [^DijkstraString$DijkstraStringDiffInfo diff-info]
  (let [a               (.getA diff-info)
        b               (.getB diff-info)
        h               (.getHead diff-info)
        normalize-index (fn [idx] (- (count a) idx))
        init-result     {:a           a
                         :deleted     {}
                         :inserted    {}
                         :levenshtein (.getDistance h)}]
    (loop [^DijkstraString$QueueElement queue-element h
           result init-result]
      (if (nil? queue-element)
        result
        (let [x (.getX queue-element)

              y (.getY queue-element)

              ^DijkstraString$Direction direction
                (.getDirection queue-element)

              ^DijkstraString$QueueElement next-queue-element
                (.previousElement queue-element)

              new-result
                (condp = direction
                  DijkstraString$Direction/diagonal
                  result

                  DijkstraString$Direction/horizontal
                  (assoc-in result
                            [:deleted (normalize-index x)]
                            (.charAt a (dec x)))

                  DijkstraString$Direction/vertical
                  (update-in result
                             [:inserted (normalize-index x)]
                             #(cons (.charAt b (dec y)) (or % '()) ))

                  DijkstraString$Direction/start
                  result)]
          (recur next-queue-element new-result))))))

(defn edit-distance
  "Convenience method"
  [a b]
  (-> (DijkstraString/diff a b) post-process  :levenshtein))


(defn diff
  "Convenience method"
  [a b]
  (let [diff-info (DijkstraString/diff a b)]
    (-> diff-info
        post-process
        (format/frmt {}))))


