;;; Copyright ©️ Rachel Bowyer 2022, 2023. All rights reserved.
;;;
;;; This program and the accompanying materials
;;; are made available under the terms of the Eclipse Public License v2.0
;;; which accompanies this distribution, and is available at
;;; https://www.eclipse.org/legal/epl-2.0/
;;;
;;; This Source Code may also be made available under the following
;;; Secondary Licenses when the conditions for such availability set forth
;;; in the Eclipse Public License, v. 2.0 are satisfied: GNU General Public
;;; License as published by the Free Software Foundation, either version 2
;;; of the License, or (at your option) any later version, with the GNU
;;; Classpath Exception which is available at
;;; https://www.gnu.org/software/classpath/license.html.


(ns info.bowyer.sam-diff.formatter.short-form
  (:require
    [clojure.string :as str]
    [info.bowyer.sam-diff.colours :as colours]
    [info.bowyer.sam-diff.formatter.string :as formatter.string]))

(def ^:private type->str
  {:inserted "Insert"
   :deleted "Delete"
   :modified "Modify"})

(def ^:private type->colour
  {:inserted colours/blue
   :deleted colours/red
   :modified colours/reset})

(defn- path->str
  [path]
  (str/join " " path))


(defn- layout-item-compare-impl
  [[a-h & a-r :as _a] [b-h & b-r :as _b]]
  (cond
    (and (nil? a-h) (nil? b-h)) 0
    (nil? a-h)                  -1
    (nil? b-h)                  1
    (not= a-h b-h)              (compare a-h b-h)
    :else                       (recur a-r b-r)))

(defn- layout-item-compare
  [a b]
  (layout-item-compare-impl (:path a) (:path b)))


(defmulti layout (fn [_path _opts diff-info] (:type diff-info)))

(defn- layout-map-seq
  [path opts {:keys [a deleted inserted modified] :as _diff-info}]
  ;; The diff engine indexes from the right but format it as indexing
  ;; from the left.
  ;; Insert is tricky as it is to the right of the element coming from the
  ;; diff engine. Formatted it is to the left of the element. So we need to
  ;; adjust for that.
  (let [normalize-key (fn [k correct] (cond->> k
                                               (number? k)
                                               (- (count a) correct)))

        create-layout (fn [type correct] (fn [[k v]] {:path (conj path (normalize-key k correct))
                                                      :type type
                                                      :value v}))
        modified      (mapcat (fn [[k v]] (layout (conj path (normalize-key k 1)) opts v))
                              modified)

        deleted       (map (create-layout :deleted 1) deleted)
        inserted      (map (create-layout :inserted 0) inserted)]
    (concat deleted inserted modified)))

(defmethod layout :atom
  [path _opts {:keys [deleted inserted] :as _diff-info}]
  [{:path  path
    :type  :modified
    :value (format "deleted %s and inserted %s" (str deleted) (str inserted))}])

(defmethod layout :string
  [path opts diff-info]
  [{:path  path
    :type  :modified
    :value (formatter.string/frmt diff-info opts)}])

(defmethod layout :set
  [path _opts {:keys [deleted inserted] :as _diff-info}]
  (let [create-layout (fn [type] (fn [v] {:path path
                                          :type type
                                          :value v}))]
    (concat (map (create-layout :deleted) deleted)
            (map (create-layout :inserted) inserted))))

(defmethod layout :map
  [path opts diff-info]
  (layout-map-seq path opts diff-info))

(defmethod layout :seq
  [path opts diff-info]
  (layout-map-seq path opts diff-info))

(defn- render
  [{frmt :format :as _opts} diffs]
  (str/join "\n"
            (for [{:keys [path type value] :as _diff} diffs]
              (if (= frmt :colour)
                (format "%s%-10s %-20s %s%s"
                        (type->colour type)
                        (type->str type)
                        (path->str path)
                        value
                        colours/reset)
                (format "%-10s %-20s %s"
                        (type->str type)
                        (path->str path)
                        value)))))

(defn frmt
  [diff-info opts]
  (->> diff-info
       (layout [] opts)
       (sort layout-item-compare)
       (render opts)))




