(ns sam-diff.optimised.dijkstra-string
  "Wrapper functions around a Dijksta based string comparison.
   It is a simple algorithm that is relatively easy to see is correct.
   It forms the basis of the more complicated variable weight bi-directional
   Dijksta non-atomic sequence algorithm.
   It is also used as a comparison in property based testing the more complicated
   algorithms."
  ;(:require [sam-diff.optimised.format-string :as format])
  (:require [sam-diff.formatter.string :as format])
  (:import [rachbowyer DijkstraString DijkstraString$DijkstraStringDiffInfo DijkstraString$QueueElement
                       DijkstraString$Direction]))

(defn- post-process
  [^DijkstraString$DijkstraStringDiffInfo diff-info a b]
  (let [h               (.getHead diff-info)
        normalize-index (fn [idx] (- (count a) idx))
        init-result     {:deleted     {}
                         :inserted    {}
                         :levenshtein (.getDistance h)}]
    (loop [^DijkstraString$QueueElement queue-element h
           result init-result]
      (if (nil? queue-element)
        result
        (let [x (.getX queue-element)

              y (.getY queue-element)

              ^DijkstraString$Direction direction
                (.getDirection queue-element)

              ^DijkstraString$QueueElement next-queue-element
                (.previousElement queue-element)

              new-result
                (condp = direction
                  DijkstraString$Direction/diagonal
                  result

                  DijkstraString$Direction/horizontal
                  (assoc-in result
                            [:deleted (normalize-index x)]
                            (.charAt a (dec x)))

                  DijkstraString$Direction/vertical
                  (update-in result
                             [:inserted (normalize-index x)]
                             #(cons (.charAt b (dec y)) (or % '()) ))

                  DijkstraString$Direction/start
                  result)]
          (recur next-queue-element new-result))))))

(defn edit-distance
  "Convenience method"
  [a b]
  (-> (DijkstraString/diff a b) (post-process a b) :levenshtein))


(defn diff
  "Convenience method"
  [a b]
  (let [diff-info (DijkstraString/diff a b)]
    (-> diff-info
        (post-process a b)
        (assoc :a a)
        (format/frmt {}))))


;(diff/diff "hell" "ello")
;=> {:levenshtein 2, :inserted {0 [(\o)]}, :type :string, :deleted {3 (\h)}, :a "hell"}

;(time (println (diff (-> "a.html" io/resource slurp)
;                     (-> "b.html" io/resource slurp))))
;{:deleted {11286 [7]}, :inserted {46525 [[&]], 46524 [[>]], 11286 [[6]]}}
;"Elapsed time: 1557.768208 msecs"

;(time (println (diff (-> "a.html" io/resource slurp)
;                     (-> "b.html" io/resource slurp))))
;{:deleted {38319 [&], 11286 [7]}, :inserted {11286 [[6]]}}
;"Elapsed time: 939.6115 msecs"

;(require '[clj-diff.core :as clj-diff])
;(time (println (diff (-> "a.html" io/resource slurp)
;                     (-> "b.html" io/resource slurp))))
;"Elapsed time: 63.384666 msecs"


;(time (edit-distance (-> "a.html" io/resource slurp)
;                     (-> "b.html" io/resource slurp)))
;"Elapsed time: 877.48175 msecs"
;=> 3


;(def a (-> "a.html" io/resource slurp))
;=> #'user/a
;(def b   (-> "b.html" io/resource slurp))
;=> #'user/b
;(time (clj-diff/edit-distance a b))
;"Elapsed time: 1.61 msecs"
;=> 3
;(time (clj-diff/edit-distance a b))
;"Elapsed time: 2.972625 msecs"
;=> 3
;(time (clj-diff/edit-distance a b))
;"Elapsed time: 1.467334 msecs"
;=> 3
