(ns hara.load
  (:refer-clojure :exclude [load]))

(defn double-vec? [x]
  (and (vector? x)
       (= 1 (count x))
       (vector? (first x))))

(defn gen-swap-func [data config]
  (fn [x]
    (cond (= '% x) data
          (double-vec? x) (list 'get-in config (first x))
          :else x)))

(defn create-load-func [form]
  (let [data (gensym)
        config  (gensym)
        rform    (clojure.walk/prewalk (gen-swap-func data config) form)]
    `(fn [~data ~config] ~rform)))

(defn load-func [loader]
  (let [f (if (:fn loader)
            (eval (create-load-func (:fn loader)))
            identity)]
    f))

(defmulti load-data  (fn [loader] (:type loader)))

(defmethod load-data :file  [loader]
  (read-string (slurp (:file loader))))

(defmethod load-data :input [loader]
  (:input loader))

(defmethod load-data :default [loader]
  (throw (Exception. (str "TYPE " (:type loader) " NOT IMPLEMENTED"))))

(defn load-key [m [k loader]]
  (let [func (load-func loader)
        data (load-data loader)]
    (assoc m k (func data m))))

(defn load-keys [m [pair & more]]
  (if (nil? pair) m
      (recur (load-key m pair) more)))

(defn load
  ([cfg] (load cfg :load))
  ([cfg start-key]
     (load-keys (dissoc cfg start-key) (get cfg start-key))))

(comment
  (load {:load [[:hello {:type :input
                         :input "hello"
                         :fn '(str % " " [[:name]])}]]
         :name "world"})
  => {:hello "hello world", :name "world"})
