;; This Source Code Form is subject to the terms of the Mozilla Public
;; License, v. 2.0. If a copy of the MPL was not distributed with this
;; file, You can obtain one at http://mozilla.org/MPL/2.0/

(ns hydrogen.core.duct-template
  (:require [clojure.string :as str]
            [hydrogen.utils :refer [resource ns->js-ns]]))

(defn- gen-cascading-routes [project-ns routes-refs]
  (as-> routes-refs $
    (map #(format "#ig/ref :%s.%s" project-ns %) $)
    (str/join "\n   " $)
    (str "\n  [" $ "]")))

(defn- use-sessions? [profiles]
  (some #(re-matches #":hydrogen/session\..*" (str %)) profiles))

(defn profile [{:keys [project-ns profiles]}]
  {:vars {:hydrogen-core? true
          :js-namespace (ns->js-ns project-ns)
          :cascading-routes (gen-cascading-routes project-ns ["static/root"
                                                              "api/config"
                                                              "api/example"])}
   :deps '[[cljs-ajax "0.8.0"]
           [day8.re-frame/http-fx "0.1.6"]
           [duct/compiler.sass "0.2.1"]
           [org.clojure/clojurescript "1.10.339"]
           [re-frame "0.10.7"]
           [reagent "0.8.1"]
           [secretary "1.2.3"]
           [hydrogen/module.core "0.1.8"]]
   :dev-deps '[[day8.re-frame/re-frame-10x "0.3.7"]]
   :templates {;; Client
               "src/{{dirs}}/client.cljs" (resource "core/cljs/client.cljs")
               "src/{{dirs}}/client/externs.js" (resource "core/cljs/externs.js")
               "src/{{dirs}}/client/home.cljs" (resource "core/cljs/home.cljs")
               "src/{{dirs}}/client/routes.cljs" (resource "core/cljs/routes.cljs")
               "src/{{dirs}}/client/session/oidc_sso.cljs" (resource "session/oidc_sso.cljs")
               "src/{{dirs}}/client/theme.cljs" (resource "core/cljs/theme.cljs")
               "src/{{dirs}}/client/util.cljs" (resource "core/cljs/util.cljs")
               "src/{{dirs}}/client/view.cljs" (resource "core/cljs/view.cljs")
               ;; API
               "src/{{dirs}}/api/config.clj" (resource "core/api/config.clj")
               "src/{{dirs}}/api/example.clj" (resource "core/api/example.clj")
               "src/{{dirs}}/api/util.clj" (resource "core/api/util.clj")
               "src/{{dirs}}/api/responses.clj" (resource "core/api/responses.clj")
               ;; Static
               "src/{{dirs}}/static/root.clj" (resource "core/static/root.clj")
               ;; Utils
               "src/{{dirs}}/util.clj" (resource "core/util.clj")
               ;; Resources
               "resources/{{dirs}}/index.html" (resource "core/resources/index.html")
               "resources/{{dirs}}/public/images/hydrogen-logo-white.svg" (resource "core/resources/images/hydrogen-logo-white.svg")
               "resources/{{dirs}}/public/css/button.scss" (resource "core/resources/css/button.scss")
               "resources/{{dirs}}/public/css/main.scss" (resource "core/resources/css/main.scss")
               "resources/{{dirs}}/public/css/theming.scss" (resource "core/resources/css/theming.scss")
               "resources/{{dirs}}/public/css/utils.scss" (resource "core/resources/css/utils.scss")}
   :modules {:hydrogen.module/core {}}
   :profile-base {:duct.middleware.web/defaults " {:security {:anti-forgery false}}"
                  :duct.middleware.web/format " {}"
                  :duct.handler/root " {:middleware [#ig/ref :duct.middleware.web/format]}"
                  :duct.compiler/sass "\n  {:source-paths [\"resources\"]\n   :output-path \"target/resources\"}"
                  (keyword (str project-ns ".static/root")) " {}"
                  (keyword (str project-ns ".api/example")) (if (use-sessions? profiles)
                                                              " {:auth-middleware #ig/ref :duct.middleware.buddy/authentication}"
                                                              " {}")
                  (keyword (str project-ns ".api/config")) " {}"}
   :dirs ["src/{{dirs}}/boundary/adapter"
          "src/{{dirs}}/boundary/port"
          "src/{{dirs}}/service"
          "src/{{dirs}}/domain"]
   :repl-options {:host "0.0.0.0"
                  :port 4001}})
