(ns hub.util.rethink
  "Helpers for RethinkDB. Candidates for a shared library if other
  services start to use Rethink."
  (:require [clojure.string :as str]
            [com.stuartsierra.component :as c]
            [environ.core :as e]
            [rethinkdb.core :as rc]
            [rethinkdb.query :as r]
            [rethinkdb.query-builder :refer [term]]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]
           [java.util UUID]))

(s/defschema RethinkSpec
  {(s/optional-key :host) s/Str
   (s/optional-key :port) s/Int
   (s/optional-key :token) s/Str
   (s/optional-key :auth-key) s/Str
   (s/optional-key :db-name) (s/named s/Str "For User service only.")})

(defonce conf
  (atom {}))

(defn db-name
  "Returns the db-name for the current connection. (This assumes only
  one database is in use.)"
  []
  (:db-name @conf))

(defn connect
  "Same as redthinkdb.core/connect, except with our configs passed
  in. Configs should be initialized via a lifecycled compoment."
  []
  (apply rc/connect (apply concat @conf)))

(defn run
  "Runs a command wrapped in a connection using conf."
  [command]
  (with-open [c (connect)]
    (r/run command c)))

(s/defn args
  [x]
  (term :ARGS [x]))

(s/defn try-create
  "Attempts to create a database with the supplied name. If it already
  exists, returns true; otherwise calls db-create."
  [db-name :- s/Str]
  (r/branch (r/contains (r/db-list) db-name)
            true
            (r/db-create db-name)))

(s/defn try-table
  "Attempts to create a table with the supplied name in the supplied
  database. If it already exists, returns true; otherwise calls
  table-create."
  [db-name :- s/Str
   table-name :- s/Str]
  (let [db (r/db db-name)]
    (r/branch (-> (r/table-list db)
                  (r/contains table-name))
              true
              (r/table-create db table-name))))

(defn get-by-id
  "Gets the document at the given id (primary key) in the table."
  [table id]
  (run (r/get table id)))

(s/defn get-field-in
  "Nested version of get-field."
  [row
   ks :- [(s/either s/Keyword s/Str)]]
  (reduce r/get-field row ks))

(letfn [(idx-exists? [table idx-name]
          (-> (r/index-list table)
              (r/contains idx-name)))]
  (defn maybe-create-index
    [index-name table index-fn]
    (run
      (r/branch (idx-exists? table index-name)
                true
                (r/index-create table index-name index-fn)))))

(defn test-database-name []
  (-> (str (UUID/randomUUID))
      (str/replace "-" "")))

(s/defschema Mode
  (s/enum :test :live))

(s/defn rethinkdb :- Lifecycle
  "Returns a Lifecycle component for the RethinkDB components."
  ([mode :- Mode
    setup-fn :- (s/=> s/Any s/Str Mode)]
   (rethinkdb mode "racehub" {} setup-fn))
  ([mode :- Mode
    db-name :- s/Str
    spec :- RethinkSpec
    setup-fn :- (s/=> s/Any s/Str Mode)]
   (let [db-name (if (= mode :test)
                   (test-database-name)
                   db-name)]
     (reify c/Lifecycle
       (start [this]
         (println "Starting RethinkDB...")
         (reset! conf (assoc spec :db-name db-name))
         (setup-fn db-name mode)
         this)
       (stop [this]
         (println "Stopping RethinkDB...")
         (when (= mode :test)
           (run (r/db-drop db-name)))
         (reset! conf nil)
         this)))))

;; ## Testing Utilities

(defn with-db
  "Create a test database, and run the given f."
  [setup-fn]
  (fn [f]
    (let [client (rethinkdb :test setup-fn)]
      (c/start client)
      (f)
      (c/stop client))))
