(ns hub.user.client
  "User Service client."
  (:require [clj-time.core :as ct]
            [clojure.string :as str]
            [clojure.core.async :as a]
            [com.stuartsierra.component :as c]
            [hub.user.api.schema :as as]
            [hub.user.client.util :refer [publish!]]
            [hub.user.facebook :as fb]
            [hub.user.oauth :as oauth]
            [hub.user.schema :as us]
            [hub.user.service :as u]
            [hub.user.setup :refer [setup!]]
            [hub.user.transforms :as t]
            [hub.util.api :as ua]
            [hub.util.rethink :as ur :refer [RethinkSpec]]
            [hub.user.util :as su :refer [encrypt compare-hash unix-time]]
            [hub.queue.client :as qc]
            [schema.core :as s])
  (:import [java.util UUID]
           [com.stuartsierra.component Lifecycle]))

;; ## Create

(s/defn create! :- (s/either as/User ua/Err)
  "Generates a user off of the supplied email and password. Email must be unique."
  [m :- as/SignupFields]
  (if (s/check as/SignupFields m)
    {:error {:type "invalid-input"}}
    (let [created (u/create! :signup m)]
      (if (false? (:ok created))
        {:error {:type (:reason created)}}
        (publish! "user-created" (t/fulluser-model->fulluser-api created))))))

(s/defn create-pending! :- (s/either as/PendingUser ua/Err)
  "Generates a pending user instance. Returns an error if a user
  exists with the supplied email."
  [m :- as/PendingUserInput]
  (if (s/check as/PendingUserInput m)
    {:error {:type "invalid-input"}}
    (let [created (t/pendinguser-model->pendinguser-api (u/create! :pending m))]
      (publish! "user-created" created))))

;; ## Facebook
;;
;; These are pretty easy to extend to the other oauth providers.

(declare get-user)

(s/defn register-facebook! :- (s/either as/FullUser ua/Err)
  "Returns a user off of the supplied facebook information. Optionally
   takes a UserID to associate the FaceBook info with.

  - If a user already exists with that facebook id, upgrades their
    token.

  - errors if another user is using the facebook account's email and
    ALREADY has another linked facebook account.

  - otherwise, associates this facebook account with that user and
    merges the new facebook info into that user's profile.

  - else, generates a new user and populates their profile using the
    supplied facebook info.

  (see fb/get-or-create-via-facebook! for impl details)"
  [m :- {:access-token s/Str
         :facebook-id s/Str
         (s/optional-key :id) as/ID}]
  (if-let [id (:id m)]
    (if-let [user (u/get-user-by-id id)]
      (-> (fb/attach-facebook! user m)
          (fb/attempt "attach-error"))
      {:error {:type "user-not-found"}})
    (-> (fb/get-or-create-via-facebook! m)
        (fb/attempt "facebook-error"))))

(s/defn unregister-facebook! :- as/User
  "Removes Facebook info from the supplied user. Returns that user."
  [id :- as/ID]
  (if-let [user (u/get-user-by-id id)]
    (t/fulluser-model->fulluser-api
     (oauth/unregister! user :facebook))
    {:error {:type "user-not-found"}}))

;; ## Get

(def LookupValue
  s/Any)

(def lookup-fns
  {:id u/get-users-by-ids
   :email u/get-users-by-emails
   :username u/get-users-by-usernames
   :name u/get-users-by-names})

(s/defn ^:private get-for-type :- {LookupValue [as/User]}
  "Returns all matches after doing a search of LookupType for the given vals."
  [type :- as/LookupType
   vals :- [LookupValue]]
  (let [{:keys [full pending]} ((lookup-fns type) vals)
        ;;:full and :pending each always have
        ;;{lookupvalue user} OR {lookupvalue [user]}
        convert-users (fn [type users]
                        (map (if (= type :full)
                               t/fulluser-model->fulluser-api
                               t/pendinguser-model->pendinguser-api)
                             users))
        transform (fn [[lookup-v user] type]
                    [lookup-v (convert-users type (if (or (list? user) (vector? user))
                                                    user
                                                    [user]))])
        ;;fulls and pendings are [lookup-v [converted-users]]
        fulls  (map transform full (repeat :full))
        pendings  (map transform pending (repeat :pending))]
    (merge-with concat (into {} fulls) (into {} pendings))))

(s/defn multiget :- {LookupValue [as/User]}
  "Returns the collection of users that mach the given query. Email,
  username, and name lookups are downcased - so returned LookupValues
  are downcased for these types as well. Should NOT use the same
  LookupValue across multiple types, or this will
  non-deterministically pick one."
  ([ms :- [{:type as/LookupType
            :value LookupValue}]]
   (let [type->vals (reduce (fn [res {:keys [type value]}]
                              (assoc res type (conj (get res type []) value)))
                            {} ms)
         ;;results is a sequence of lookup-v->users
         results   (map (fn [[type vals]]
                          (get-for-type type vals))
                        type->vals)]
     (apply merge results)))

  ([type :- as/LookupType
    vals :- [LookupValue]]
   (multiget (for [v vals] {:type type :value v}))))

(s/defn get-user :- (s/maybe as/User)
  ([id :- as/ID] (get-user :id id))
  ([type :- (s/enum :id :email :username :name :facebook-id)
    v :- LookupValue]
   (if (= type :facebook-id)
     (t/fulluser-model->fulluser-api
      (u/get-user-by-fb-id v))
     (-> (multiget [{:type type :value v}])
         (get v)
         (first)))))

(s/defn search :- {:full [as/FullUser]
                   :pending [as/PendingUser]}
  "Can search by first name, last name, full name, username, or
  email. Will do a downcased, whitespace-removed search."
  [query :- s/Str]
  {:full (map t/fulluser-model->fulluser-api (u/search-full query))
   :pending (map t/pendinguser-model->pendinguser-api
                 (u/search-pending query))})

;; ## Update
(s/defn ^:private build-profile :- (s/either as/Profile as/PendingProfile)
  "Creates a profile update."
  [{:keys [first-name last-name] :as m} :- as/UserUpdate]
  (merge (select-keys m [:gender :phone :location :birthdate :photos])
         (when-let [name-map (merge (when first-name
                                      {:first first-name})
                                    (when last-name
                                      {:last last-name}))]
           {:name name-map})))

(s/defn ^:private build-update :- {s/Keyword s/Any}
  "Builds an update for the given user, from the UserUpdate. Returns
  the map to update! into the user."
  [user-to-update :- us/User
   user-type :- (s/enum :full :pending)
   {:keys [name username password email-address] :as m} :- as/UserUpdate]
  (let [full? (= user-type :full)
        profile-updates (build-profile m)]
    (merge (when (not-empty profile-updates)
             {:profile profile-updates})
           ;; Pending users only:
           (when (and (not full?) (not-empty name))
             {:name (:name m)})
           ;; Full users only:
           (when (and full? (not-empty username))
             {:username (:username m)})
           (when (and full? (not-empty password))
             {:password (encrypt password)})
           ;; Email:
           (when (not-empty email-address)
             {:email {:address email-address
                      :verified?
                      (let [update-user-email (-> user-to-update
                                                  :email
                                                  :address)]
                        (if (and (not-empty update-user-email)
                                 (= (str/lower-case email-address)
                                    (str/lower-case update-user-email)))
                          (:verified? (:email user-to-update))
                          false))}}))))

(s/defn update! :- (s/either as/User ua/Err)
  "Updates the user. If a new value for password, email-address or a
  first or last name is passed in, updates those fields
  appropriately (password gets hashed). Errors if the user doesn't
  exist, otherwise returns the user with updates."
  ;;TODO: Transmit changes on queue (only if email got changed? - so
  ;;mailer can send verification.)
  [id :- as/ID
   {:keys [username email-address] :as m} :- as/UserUpdate]
  (if (s/check as/UserUpdate m)
    {:error {:type "invalid-input"}}
    (let [{:keys [full pending]} (u/get-users-by-ids [id])]
      (if (or (not-empty full) (not-empty pending))
        (let [user-type (if (not-empty full)
                          :full
                          :pending)
              user-to-update (or (get full id) (get pending id))
              result (->> (build-update user-to-update user-type m)
                          (u/update! id))]
          (if (false? (:ok result))
            {:error (if (= (:reason result) "Email is already taken!")
                      {:type "email-taken"}
                      {:type "username-taken"})}
            (let [api-user (if (= user-type :full)
                             (t/fulluser-model->fulluser-api result)
                             (t/pendinguser-model->pendinguser-api result))]
              (publish! "user-updated" api-user))))
        {:error {:type "user-not-found"}}))))

(s/defn put-profile! :- (s/either as/User ua/Err)
  "Put the given user-doc (replace the old one). Cant change id. Use
  this if you want to dissoc a field."
  [id :- s/Str
   profile :- as/Profile]
  (let [{:keys [full pending]} (u/get-users-by-ids [id])
        full-user (get full id)
        pending-user (get pending id)
        invalid-profile? #(s/check as/Profile %)
        put-user! (fn [user transform]
                    (-> (assoc user :profile profile)
                        u/put!
                        transform))]
    (cond
      (invalid-profile? profile) {:error {:type "invalid-input"}}
      full-user (let [u (put-user! full-user t/fulluser-model->fulluser-api)]
                  (publish! "user-updated" u))
      pending-user (let [u (put-user! pending-user t/pendinguser-model->pendinguser-api)]
                     (publish! "user-updated" u))
      :else {:error {:type "user-not-found"}})))

(s/defn delete! :- (s/either {:deleted (s/eq true)}
                             ua/Err)
  "Returns true if delete succeeded, false otherwise."
  [id :- as/ID]
  (if (u/delete-by-id! id)
    {:deleted true}
    {:error {:type "delete-failed"}}))

;; ## Authentication

(s/defn trigger-password-reset! :- (s/either s/Int ua/Err)
  "Generate a password reset and activation code for the supplied
  user. Errors if the user doesn't exist. Returns number of
  subscribers to trigger-password-reset event."
  [id :- as/ID]
  (let [reset-code (UUID/randomUUID)
        created-at (unix-time)
        {:keys [password-reset] :as pw} {:password-reset {:code reset-code
                                                          :created-at created-at}}]
    (let [u (u/update! id pw)]
      (if (false? (:ok u))
        {:error {:type "user-not-found"}}
        (qc/publish! "trigger-password-reset" {:user-id id
                                               :name (get-in u [:profile :name])
                                               :email (:address (:email u))
                                               :password-reset
                                               {:code (str (:code password-reset))
                                                :created-at (:created-at password-reset)}})))))

(def reset-code-lifespan
  ;;2 days
  (ct/in-millis (ct/days 2)))

(s/defn ^:private reset-code-expired?
  [{:keys [password-reset]} :- us/User]
  (< reset-code-lifespan
     (- (unix-time) (:created-at password-reset))))

(s/defn reset-password! :- (s/either {:user-id as/ID}
                                     ua/Err)
  "If the reset-code is valid for the giver user-id, will set save a
  hashed version of their new-password and return their user-id. Else
  returns an error."
  [reset-code :- s/Str
   new-password :- s/Str]
  (let [{:keys [password-reset] :as user} (u/get-user-by-reset-code reset-code)]
    (if (= (:code password-reset) reset-code)
      (if (reset-code-expired? user)
        {:error {:type "expired-reset-code"}}
        (if-let [user (u/put! (-> user
                                  (dissoc :password-reset)
                                  (assoc :password (encrypt new-password))))]
          {:user-id (:id user)}
          {:error {:type "error-updating-password"}}))
      {:error {:type "incorrect-reset-code"}})))

(s/defn reset-code-info :- {:state (s/enum :valid :expired :invalid)
                            (s/optional-key :user-id) as/ID}
  "Returns the current state of the pw reset code, and if it's a valid
  code (possibly expired), then the user-id to whom it belongs."
  [reset-code :- s/Str]
  (if-let [user (u/get-user-by-reset-code reset-code)]
    {:user-id (:id user)
     :state (if (reset-code-expired? user)
              :expired
              :valid)}
    {:state :invalid}))

(s/defn password-valid? :- (s/either s/Bool ua/Err)
  "Does the user with the supplied ID's password match this password?
  Errors if the password isn't set."
  [id :- as/ID
   password :- s/Str]
  (if-let [user (get (:full (u/get-users-by-ids [id])) id)]
    (if (not-empty (:password user))
      (compare-hash password (:password user))
      {:error {:type "password-not-set"}})
    {:error {:type "user-not-found"}}))

;; ## User Merge

(s/defn merge-users! :- (s/either as/User ua/Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge full users, OR to merge pending users
  together or into full users. Can't merge full users into a pending
  user."
  [primary-id :- as/ID
   sub-id :- as/ID]
  ;;TODO: Put something on the queue when this happens. Think about
  ;;coordination with other services. Ideally we would know if an
  ;;error occured in any service's merge-user implementation.
  (if-let [merged (u/merge-users! primary-id sub-id)]
    (let [result (if (= (u/user-type merged) :full)
                   (t/fulluser-model->fulluser-api merged)
                   (t/pendinguser-model->pendinguser-api merged))]
      (qc/publish! "user-merge" {:primary-user primary-id
                                 :absorbed-user sub-id})
      result)
    {:error {:type "merge-user-error"}}))

;; ## Client Startup
(defonce system (atom nil))

(s/defschema UserClientOpts
  {:rethink {:mode (s/enum :test :live)
             :db s/Str
             :spec RethinkSpec}
   :queue {:spec qc/RedisSpec}})

(s/defn user-client
  "Client for the user service. Returns lifecycled components for the
  datastore and event queue."
  ([] (user-client {:rethink {:mode :live
                              :db "racehub"
                              :spec {}}
                    :queue {:spec {}}}))
  ([{:keys [rethink queue] :as opts} :- UserClientOpts]
   (let [{:keys [mode db spec]} rethink]
     (c/system-map :rethink (ur/rethinkdb mode db spec setup!)
                   :queue (qc/queue-client (:spec queue))))))

(s/defn startup!
  "Starts the user client"
  []
  (reset! system (user-client))
  (c/start @system))

(defn stop!
  "Stops the user client."
  []
  (c/stop @system)
  (reset! system nil))
