(ns hub.user.oauth
  (:require [hub.user.schema :refer [User]]
            [hub.user.service :as user]
            [hub.user.transforms :as trans]
            [schema.core :as s]))

;; ## OAuthProvider Manipulations

(defmulti oauth-metadata
  "Grabs relevant metadata for the linked oauth service."
  (fn [auth-type token] auth-type))

(def OAuthProvider s/Keyword)

(s/defmethod oauth-metadata :default
  [auth-type :- OAuthProvider token :- s/Str]
  {})

(defmulti oauth-name
  "Returns the user's name for the given oauth."
  (fn [auth-type metadata] auth-type))

(s/defmethod oauth-name :default :- s/Str
  [auth-type :- OAuthProvider
   metadata :- s/Any]
  "")

(s/defn register-auth! :- User
  "Adds an oauth token of the supplied type to the username."
  [user :- User auth-type :- OAuthProvider token :- s/Str id :- s/Str]
  (let [meta (oauth-metadata auth-type token)]
    (user/put! (assoc-in user [:oauth auth-type]
                         {:token token
                          :id id
                          :metadata meta}))))

(s/defn unregister! :- User
  "Removes the supplied registration type from the user."
  [user :- User auth-type :- OAuthProvider]
  (user/put! (update user :oauth dissoc auth-type)))

(s/defn oauth-registered? :- s/Bool
  "Returns true if the user has a current registration for this
    oauth type, false otherwise."
  [user :- User auth-type :- OAuthProvider]
  (boolean
   (get-in user [:oauth auth-type])))
