(ns hub.user.util
  "User service helpers."
  (:require [clj-time.coerce :as tc]
            [clj-time.core :refer [now]]
            [clojure.string :as str]
            [hub.user.schema :as us]
            [rethinkdb.query :as r]
            [schema.core :as s])
  (:import [org.mindrot.jbcrypt BCrypt]))

(s/def UnixTime
  (s/named s/Int "Instant in time defined as the number of ms
  since midnight UTC Jan 1 1970."))

(s/defn unix-time :- UnixTime
  "Returns the current UnixTime"
  []
  (tc/to-long (now)))

;; ## CREATE Helpers

(s/defn username-valid? :- s/Bool
  [s :- s/Str]
  (boolean
   (re-matches #"[a-zA-Z0-9]{3,20}" s)))

(s/defn random-usernames
  "Don't put an output schema on this! Returns an infinite sequence
    of usernames created by sticking the prefix onto the beginning of
    random numbers below the supplied maximum value."
  [prefix :- s/Str
   max :- s/Int]
  (map (partial str prefix) (repeatedly (partial rand-int max))))


(s/defn select-passing-keys
  "Selects the keys in the map that pass the given schema."
  [schema m]
  (reduce (fn [res [k v]]
            (if-let [key-schema (or (get schema k)
                                    (get schema (s/optional-key k)))]
              (if (s/check key-schema (get m k))
                res
                (assoc res k v))
              res))
          {} m))

(s/defn build-profile :- us/Profile
  [data :- {s/Any s/Any}]
  (select-passing-keys us/Profile data))

(s/defn gen-salt :- s/Str
  ([size]
   (BCrypt/gensalt size))
  ([]
   (BCrypt/gensalt)))

(s/defn encrypt :- s/Str
  "Encrypt the given string with a generated or supplied salt. Uses
  BCrypt for strong hashing."
  ;; generate a salt
  ([salt raw] (BCrypt/hashpw raw salt))
  ([raw] (encrypt (gen-salt) raw)))

(s/defn compare-hash :- s/Bool
  "Compare a raw string with an already encrypted string"
  [raw encrypted]
  (BCrypt/checkpw raw encrypted))

(defn deep-merge
  "Takes in multiple maps, and returns one merged map, where all
  nested maps are merged as well."
  [& xs]
  (letfn [(merge* [l r]
            (if (and (map? l) (map? r))
              (merge-with merge* l r)
              r))]
    (reduce merge* {} xs)))
