(ns hub.user.setup
  "Setup of user service - create database, tables, indices."
  (:require  [hub.user.rethink :as ur :refer [ args]]
             [rethinkdb.query :as r]
             [schema.core :as s]))

(s/defn setup!
  "Basic setup for the user table."
  [mode :- (s/enum :test :live)]
  (let [db-name (ur/db-name)
        fulluser-table #(r/table (r/db (ur/db-name)) "fulluser")
        pendinguser-table #(r/table (r/db (ur/db-name)) "pendinguser")
        idx-exists? (fn [table idx-name]
                      (-> (r/index-list table)
                          (r/contains idx-name)))
        maybe-create-idx (fn [idx-name table idx-fun]
                           (ur/run (r/branch (idx-exists? table idx-name)
                                             true
                                             (r/index-create
                                              table
                                              idx-name
                                              idx-fun))))]
    ;;create racehub database, if needed
    (ur/run (ur/try-create db-name))

    ;;create full user table, if needed
    (ur/run (ur/try-table db-name "fulluser"))

    ;;create pending user table, if needed
    (ur/run (ur/try-table db-name "pendinguser"))

    ;;FullUser Index: Username
    (maybe-create-idx "username" (fulluser-table) (r/fn [row]
                                                    (r/downcase
                                                     (r/get-field row :username))))

    ;;FullUser Index: Facebook ID
    (maybe-create-idx "facebook-id" (fulluser-table)
                      (r/fn [row]
                        (r/downcase
                         (ur/get-field-in row [:oauth :facebook :metadata :id]))))

    ;;FullUser Index: Email Address
    (maybe-create-idx "email" (fulluser-table) (r/fn [row]
                                                 (r/downcase
                                                  (ur/get-field-in row [:email :address]))))

    ;;PendingUser Index: Email Address
    (maybe-create-idx "email"
                      (pendinguser-table)
                      (r/fn [row]
                        (r/downcase
                         (ur/get-field-in row  [:email :address]))))

    (maybe-create-idx "name"
                      (fulluser-table)
                      (r/fn [row]
                        (-> (r/add (ur/get-field-in row [:profile :name :first])
                                   (ur/get-field-in row [:profile :name :last]))
                            (r/split)
                            args
                            r/add
                            r/downcase)))

    ;;PendingUser Index: Name
    (maybe-create-idx "name" (pendinguser-table)
                      (r/fn [row]
                        (r/add (args (r/split (r/downcase (r/get-field row :name)))))))

    ;;FullUser Index: Reset Code
    (maybe-create-idx "pw-reset-code"
                      (fulluser-table)
                      (r/fn [row]
                        (ur/get-field-in row [:password-reset :code])))

    ;;FullUser Index: Facebook Email
    (maybe-create-idx "facebook-email"
                      (fulluser-table)
                      (r/fn [row]
                        (r/downcase
                         (ur/get-field-in row [:oauth :facebook :metadata :email]))))

    ;;FullUser Index: Facebook ID
    (maybe-create-idx "facebook-id"
                      (fulluser-table)
                      (r/fn [row]
                        (r/downcase
                         (ur/get-field-in row [:oauth :facebook :metadata :id]))))))
